<?php

namespace App\Models;

use App\Enums\StatusEnum;
use App\Traits\Filterable;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Support\Str;

class SocialAccount extends Model
{
    use HasFactory, Filterable;

    protected $guarded = [];

    protected $casts = [
        'account_information' => 'object',
    ];


    protected static function booted()
    {
        static::creating(function (Model $model) {
            $model->uid = Str::uuid();
        });
    }



    /**
     * User where account belongs
     *
     * @return BelongsTo
     */
    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class, "user_id");
    }


    /**
     * Platform where account belongs
     *
     * @return BelongsTo
     */
    public function platform(): BelongsTo
    {
        return $this->belongsTo(Platform::class, "platform_id");
    }


    /**
     * subscription where account belongs
     *
     * @return BelongsTo
     */
    public function workspace(): BelongsTo
    {
        return $this->belongsTo(Workspace::class, "workspace_id");
    }


    /**
     * Get social post
     *
     * @return HasMany
     */
    public function posts(): HasMany
    {
        return $this->hasMany(SocialPost::class, "account_id");
    }


    public function scopeActive(Builder $q): Builder
    {
        return $q->where('status', StatusEnum::true->status());
    }
    public function scopeInactive(Builder $q): Builder
    {
        return $q->where('status', StatusEnum::false->status());
    }


    public function scopeConnected(Builder $q): Builder
    {
        return $q->where('is_connected', StatusEnum::true->status());
    }

    public function scopeFilterByPlatform(Builder $query)
    {
        $platformSlug = request()->input('slug');

        if ($platformSlug) {
            $query->whereHas('platform', function ($q) use ($platformSlug) {
                $q->where('slug', $platformSlug);
            });
        }

        return $query;
    }
}
