"use client"

import { useState } from "react"
import { ChevronLeft, ChevronRight, TrendingUp, BarChart3 } from "lucide-react"
import useTranslate from "@/hooks/useTranslate"


const PerformanceSummary = ({ data, metrics }) => {
  const [currentIndex, setCurrentIndex] = useState(0)
  const t = useTranslate()

  const formatNumber = (num) => {
    if (typeof num === "string" && num.includes("%")) {
      return num 
    }

    const number = typeof num === "string" ? Number.parseInt(num.replace(/,/g, "")) : num

    if (number >= 1000000) {
      return (number / 1000000).toFixed(1).replace(/\.0$/, "") + "M"
    }
    if (number >= 1000) {
      return (number / 1000).toFixed(1).replace(/\.0$/, "") + "K"
    }
    return number.toString()
  }

  const processedMetrics =
    metrics?.map((metric) => ({
      ...metric,
      value: metric.value
        ? formatNumber(metric.value)
        : metric.dataKey
          ? metric.isPercentage
            ? `${data?.[metric.dataKey] || 0}%`
            : formatNumber(data?.[metric.dataKey] || 0)
          : "0",
    })) || []

  const visibleMetrics = processedMetrics.slice(currentIndex, currentIndex + 4)

  const handlePrevious = () => {
    setCurrentIndex((prev) => Math.max(0, prev - 1))
  }

  const handleNext = () => {
    setCurrentIndex((prev) => Math.min(processedMetrics.length - 4, prev + 1))
  }

  const canGoPrevious = currentIndex > 0
  const canGoNext = currentIndex < processedMetrics.length - 4

  return (
    <div className="bg-white dark:bg-zinc-900 rounded-xl border border-gray-200 dark:border-zinc-800 shadow-sm hover:shadow-md transition-shadow duration-200">

      <div className="flex items-center justify-between flex-wrap gap-4 p-6 pb-4">
        <div className="flex items-start gap-4">
          <div className="mt-2">
            <TrendingUp className="h-6 w-6 text-primary" />
          </div>
          <div>
            <h2 className="lg:text-2xl text-xl font-bold bg-gradient-to-r from-gray-900 to-gray-700 dark:from-white dark:to-gray-300 bg-clip-text text-transparent">
              {t("Performance Summary")}
            </h2>
            <p className="text-gray-600 dark:text-gray-400 text-sm mt-1">
              {t("View your key profile performance metrics from the reporting")}
            </p>
          </div>
        </div>

        {/* Navigation Arrows */}
        <div className="flex items-center gap-2">
          <button
            onClick={handlePrevious}
            disabled={!canGoPrevious}
            className={`p-3 rounded-xl border transition-colors duration-200 ${
              canGoPrevious
                ? "border-gray-200 dark:border-zinc-800 hover:bg-gray-50 dark:hover:bg-zinc-800 text-gray-600 dark:text-gray-400"
                : "border-gray-200 dark:border-zinc-800 text-gray-300 dark:text-gray-600 cursor-not-allowed"
            }`}
          >
            <ChevronLeft className="w-4 h-4" />
          </button>
          <button
            onClick={handleNext}
            disabled={!canGoNext}
            className={`p-3 rounded-xl border transition-colors duration-200 ${
              canGoNext
                ? "border-gray-200 dark:border-zinc-800 hover:bg-gray-50 dark:hover:bg-zinc-800 text-gray-600 dark:text-gray-400"
                : "border-gray-200 dark:border-zinc-800 text-gray-300 dark:text-gray-600 cursor-not-allowed"
            }`}
          >
            <ChevronRight className="w-4 h-4" />
          </button>
        </div>
      </div>

      <div className="border-t border-gray-100 dark:border-zinc-800">
        <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-4 divide-x divide-gray-200 dark:divide-zinc-700">
          {visibleMetrics.map((metric, index) => (
            <div 
              key={index} 
              className="p-6 hover:bg-gray-50/50 dark:hover:bg-zinc-800/50 transition-colors duration-200"
            >
              <div className="space-y-4">
                <div>
                  <h3 className="text-gray-900 dark:text-white font-semibold text-lg">
                    {metric.title}
                  </h3>
                  <p className="text-gray-500 dark:text-gray-400 text-sm mt-1">
                    {metric.label}
                  </p>
                </div>

                <div className="flex items-end justify-between">
                  <div className="lg:text-3xl text-2xl font-bold text-gray-900 dark:text-white">
                    {metric.value}
                  </div>
                  <div className={`flex items-center gap-1 px-2 py-1 rounded-full ${metric.color} bg-opacity-10`}>
                    <span className="text-sm font-semibold">{metric.change}</span>
                    <svg 
                      className="w-4 h-4" 
                      fill="currentColor" 
                      viewBox="0 0 20 20"
                      style={{ 
                        transform: metric.changeType === "increase" ? "rotate(0deg)" : "rotate(180deg)" 
                      }}
                    >
                      <path
                        fillRule="evenodd"
                        d="M5.293 9.707a1 1 0 010-1.414l4-4a1 1 0 011.414 0l4 4a1 1 0 01-1.414 1.414L11 7.414V15a1 1 0 11-2 0V7.414L6.707 9.707a1 1 0 01-1.414 0z"
                        clipRule="evenodd"
                      />
                    </svg>
                  </div>
                </div>
              </div>
            </div>
          ))}
        </div>
      </div>
    </div>
  )
}

export default PerformanceSummary
