"use client"

import { useState } from "react"
import { Users, FileText, TrendingUp, Grid3X3, List, Sparkles, Zap, Plus } from "lucide-react"
import { Button } from "@/Components/UI/Button"
import AppLayout from "@/Layouts/AppLayout"
import OverviewCard from "@/Components/Platform/OverviewCard"
import PlatformCard from "@/Components/Platform/PlatformCard"
import ConnectAccountModal from "@/Components/Post/ConnectAccountModal"
import { motion, AnimatePresence } from "framer-motion";
import { route } from "ziggy-js";
import useTranslate from "@/hooks/useTranslate";

export default function PlatformIndex({ platforms, overview }) {
  const t = useTranslate();
  const breadcrumbs = [{ label: "Home", href: route("user.home") }, { label: "Platforms" }]
  const [viewMode, setViewMode] = useState("grid")
  const [connectModalOpen, setConnectModalOpen] = useState(false)
  const [selectedPlatformForConnect, setSelectedPlatformForConnect] = useState(null)



  const overviewStats = [
    { title: "Accounts", value: overview.accounts, icon: Users },
    { title: "Followers", value: overview.followers, icon: Users },
    { title: "Posts", value: overview.posts, icon: FileText },
    { title: "Reached", value: overview.reach, icon: TrendingUp },
  ]

  return (
    <AppLayout breadcrumbs={breadcrumbs}>
      <div className="flex-1 p-6 overflow-auto bg-gradient-to-br from-gray-50 to-gray-100 dark:from-zinc-950 dark:to-black min-h-screen">
        {/* Header */}
        <div className="mb-6">
          <div className="flex items-center gap-3 mb-2">
            <div>
              <Zap className="h-6 w-6 text-primary" />
            </div>
            <h1 className="text-xl sm:text-2xl dark:text-gray-200 font-bold">
              Platform Overview
            </h1>
          </div>
          <p className="text-gray-500 dark:text-gray-200">
            Monitor and manage your social media presence across all platforms
          </p>
        </div>
        
        <div className="space-y-8">
          {/* Overview Cards */}
          <motion.div
            initial={{ opacity: 0, y: 20 }}
            animate={{ opacity: 1, y: 0 }}
            transition={{ duration: 0.6 }}
          >
            <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6">
              {overviewStats.map((stat, index) => (
                <OverviewCard key={index} title={stat.title} value={stat.value} icon={stat.icon} index={index} />
              ))}
            </div>
          </motion.div>

        {/* All Platforms Section */}
        <motion.div
          initial={{ opacity: 0, y: 20 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ duration: 0.6, delay: 0.2 }}
        >
          <div className="flex items-center justify-between flex-wrap gap-4 mb-8">
            <div>
              <h2 className="text-lg sm:text-xl dark:text-gray-200 font-bold">Connected Platforms</h2>
              <p className="text-gray-600 dark:text-gray-400">Manage your social media accounts and settings</p>
            </div>
            <div className="flex items-center gap-4">
              <Button
                onClick={() => setConnectModalOpen(true)}
                className="bg-gradient-to-r from-primary to-secondary text-white hover:from-secondary hover:to-primary shadow-sm hover:shadow-md transition-all duration-200"
              >
                <Plus className="w-4 h-4 mr-2" />
                Connect Account
              </Button>

              <div className="flex items-center space-x-2 bg-white dark:bg-zinc-900 rounded-lg p-1 shadow-sm border border-gray-200 dark:border-zinc-800">
              <Button
                variant="ghost"
                size="sm"
                onClick={() => setViewMode("grid")}
                className={`transition-all duration-200 ${
                  viewMode === "grid"
                    ? "bg-gradient-to-r from-primary to-secondary text-white shadow-sm"
                    : "text-gray-600 dark:text-gray-400 hover:text-gray-900 dark:hover:text-white hover:bg-gray-50 dark:hover:bg-zinc-800"
                }`}
              >
                <Grid3X3 className="h-4 w-4" />
              </Button>
              <Button
                variant="ghost"
                size="sm"
                onClick={() => setViewMode("list")}
                className={`transition-all duration-200 ${
                  viewMode === "list"
                    ? "bg-gradient-to-r from-primary to-secondary text-white shadow-sm"
                    : "text-gray-600 dark:text-gray-400 hover:text-gray-900 dark:hover:text-white hover:bg-gray-50 dark:hover:bg-zinc-800"
                }`}
              >
                <List className="h-4 w-4" />
              </Button>
              </div>
            </div>
          </div>

          <AnimatePresence mode="wait">
            <motion.div
              key={viewMode}
              initial={{ opacity: 0, y: 20 }}
              animate={{ opacity: 1, y: 0 }}
              exit={{ opacity: 0, y: -20 }}
              transition={{ duration: 0.3 }}
              className={viewMode === "grid" ? "grid grid-cols-1 lg:grid-cols-2 gap-6" : "space-y-6"}
            >
              {platforms.map((platform, index) => (
                <motion.div
                  key={platform.id}
                  initial={{ opacity: 0, y: 20 }}
                  animate={{ opacity: 1, y: 0 }}
                  transition={{ duration: 0.3, delay: index * 0.1 }}
                >
                  <PlatformCard
                    platform={platform}
                    viewMode={viewMode}
                    onAddAccount={(platform) => {
                      setSelectedPlatformForConnect(platform)
                      setConnectModalOpen(true)
                    }}
                  />
                </motion.div>
              ))}
            </motion.div>
          </AnimatePresence>

          {platforms.length === 0 && (
            <motion.div
              initial={{ opacity: 0, scale: 0.95 }}
              animate={{ opacity: 1, scale: 1 }}
              className="text-center py-16"
            >
              <div className="p-4 bg-gray-100 dark:bg-zinc-800 rounded-full inline-block mb-4">
                <Users className="h-12 w-12 text-gray-400" />
              </div>
              <h3 className="text-xl font-semibold text-gray-900 dark:text-white mb-2">No Platforms Connected</h3>
              <p className="text-gray-600 dark:text-gray-400">Start by connecting your social media accounts to get started.</p>
            </motion.div>
          )}
        </motion.div>
        </div>

        {/* Connect Account Modal */}
        <ConnectAccountModal
          isOpen={connectModalOpen}
          onClose={() => {
            setConnectModalOpen(false)
            setSelectedPlatformForConnect(null)
          }}
          platforms={platforms}
          selectedPlatform={selectedPlatformForConnect}
          t={t}
        />
      </div>
    </AppLayout>
  )
}
