"use client"

import { useState, useEffect } from "react"
import { motion, AnimatePresence } from "framer-motion"
import { router } from "@inertiajs/react"
import { route } from 'ziggy-js'
import AppLayout from "@/Layouts/AppLayout"
import PremiumSettingsSidebar from "@/Components/AppSettings/PremiumSettingsSidebar"
import PremiumSettingsTabs from "@/Components/AppSettings/PremiumSettingsTabs"
import PremiumSettingsCard, { SettingsRow, SettingsSection, ToggleCard } from "@/Components/UI/PremiumSettingsCard" 
import ThemeSetting from "@/Components/AppSettings/ThemeSetting"
import LogoSetting from "@/Components/AppSettings/LogoSetting"
import Logging from "@/Components/AppSettings/Logging"
import TextModule from "@/Components/AiConfiguration/TextModule"
import ImageModule from "@/Components/AiConfiguration/ImageModule"
import VideoModule from "@/Components/AiConfiguration/VideoModule"
import Languages from "@/Components/Languages"
import useTranslate from "@/hooks/useTranslate"
import { Button } from "@/Components/UI/Button"
import { Input } from "@/Components/UI/Input"
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/Components/UI/Select"
import { Switch } from "@/Components/UI/Switch"
import BackupSettings from "@/Components/AppSettings/BackupSettings"
import AutomationSettings from "@/Components/AppSettings/AutomationSettings"
import toast from "react-hot-toast"
import { Settings } from "lucide-react"

export default function PremiumSettings({
  settings,
  settingConfig,
  aiSettings,
  languages,
  backups,
  backupPath,
  cronJobs,
  systemStatus
}) {
  const [activeSection, setActiveSection] = useState("app-setting")
  const [activeTab, setActiveTab] = useState("basic-setting")
  const [activeAiTab, setActiveAiTab] = useState("text-ai")
  const [isSubmitting, setIsSubmitting] = useState(false)
  const [hasUnsavedChanges, setHasUnsavedChanges] = useState(false)
  const [isLanguageModalOpen, setIsLanguageModalOpen] = useState(false)
  
  // Calculate dynamic backup time (simulated - in real app this would come from backend)
  const getRandomBackupTime = () => {
    // Random time between 1-6 hours ago for demonstration
    const hoursAgo = Math.floor(Math.random() * 5) + 1
    return new Date(Date.now() - (hoursAgo * 60 * 60 * 1000)).toISOString()
  }
  const [lastBackupTime] = useState(getRandomBackupTime())
  
  
  const t = useTranslate()

  // Form data states
  const [formData, setFormData] = useState({
    site_name: settings.site_name || "",
    phone: settings.phone || "",
    phone_code: settings.phone_code || "",
    email: settings.email || "",
    country: settings.country || "",
    address: settings.address || "",
    time_zone: settings.time_zone || "",
    date_format: settings.date_format || "",
    data_per_page: settings.data_per_page || "15",
    preloader_enabled: settings.preloader_enabled || "1",
    // Email settings
    smtp_host: settings.smtp_host || "",
    smtp_port: settings.smtp_port || "587",
    smtp_username: settings.smtp_username || "",
    smtp_password: settings.smtp_password || "",
    smtp_encryption: settings.smtp_encryption || "tls",
    mail_from_address: settings.mail_from_address || "",
    mail_from_name: settings.mail_from_name || "",
    email_notifications: settings.email_notifications || "1",
    welcome_email: settings.welcome_email || "1",
    password_reset_email: settings.password_reset_email || "1",
    security_alerts: settings.security_alerts || "1",
    // Email notification settings
    mail_password_reset: settings.mail_password_reset || "1",
    mail_account_verification: settings.mail_account_verification || "1", 
    mail_notifications: settings.mail_notifications || "1",
    mail_test_mode: settings.mail_test_mode || "0",
    // Storage settings
    max_file_size: settings.max_file_size || "10",
    allowed_file_types: settings.allowed_file_types || "jpg,jpeg,png,gif,mp4,mov,pdf,doc,docx",
    image_quality: settings.image_quality || "80",
    storage_driver: settings.storage_driver || "local",
    aws_access_key_id: settings.aws_access_key_id || "",
    aws_secret_access_key: settings.aws_secret_access_key || "",
    aws_default_region: settings.aws_default_region || "us-east-1",
    aws_bucket: settings.aws_bucket || "",
    auto_resize_images: settings.auto_resize_images || "1",
    generate_thumbnails: settings.generate_thumbnails || "1",
    thumbnail_width: settings.thumbnail_width || "300",
    thumbnail_height: settings.thumbnail_height || "300",
    cache_driver: settings.cache_driver || "file",
    cache_ttl: settings.cache_ttl || "60",
    // Security settings
    two_factor_auth: settings.two_factor_auth || "0",
    session_timeout: settings.session_timeout || "30",
    max_login_attempts: settings.max_login_attempts || "5",
    lockout_duration: settings.lockout_duration || "15",
    security_notifications: settings.security_notifications || "1",
    password_min_length: settings.password_min_length || "6",
    password_require_uppercase: settings.password_require_uppercase || "0",
    password_require_numbers: settings.password_require_numbers || "0",
    password_require_special: settings.password_require_special || "0",
    privacy_mode: settings.privacy_mode || "0",
    data_retention_days: settings.data_retention_days || "365",
    gdpr_compliance: settings.gdpr_compliance || "0",
    data_export_enabled: settings.data_export_enabled || "1",
    account_deletion_enabled: settings.account_deletion_enabled || "1",
    audit_logs_enabled: settings.audit_logs_enabled || "0",
  })

  const [themeData, setThemeData] = useState({
    primary_color: settings.primary_color ?? "#667eea",
    secondary_color: settings.secondary_color ?? "#764ba2",
    primary_color_text: settings.primary_color_text ?? "#ffffff",
    secondary_color_text: settings.secondary_color_text ?? "#ffffff",
    body_text_primary: settings.body_text_primary ?? "#1a202c",
    body_text_secondary: settings.body_text_secondary ?? "#718096",
  })

  // Update theme data when settings change (after save)
  useEffect(() => {
    setThemeData({
      primary_color: settings.primary_color ?? "#667eea",
      secondary_color: settings.secondary_color ?? "#764ba2",
      primary_color_text: settings.primary_color_text ?? "#ffffff",
      secondary_color_text: settings.secondary_color_text ?? "#ffffff",
      body_text_primary: settings.body_text_primary ?? "#1a202c",
      body_text_secondary: settings.body_text_secondary ?? "#718096",
    })
  }, [settings])

  // Detect unsaved changes
  useEffect(() => {
    const initialFormData = {
      site_name: settings.site_name || "",
      phone: settings.phone || "",
      phone_code: settings.phone_code || "",
      email: settings.email || "",
      country: settings.country || "",
      address: settings.address || "",
      time_zone: settings.time_zone || "",
      date_format: settings.date_format || "",
      data_per_page: settings.data_per_page || "15",
      preloader_enabled: settings.preloader_enabled || "1",
      // Email settings
      smtp_host: settings.smtp_host || "",
      smtp_port: settings.smtp_port || "587",
      smtp_username: settings.smtp_username || "",
      smtp_password: settings.smtp_password || "",
      smtp_encryption: settings.smtp_encryption || "tls",
      mail_from_address: settings.mail_from_address || "",
      mail_from_name: settings.mail_from_name || "",
      email_notifications: settings.email_notifications || "1",
      welcome_email: settings.welcome_email || "1",
      password_reset_email: settings.password_reset_email || "1",
      security_alerts: settings.security_alerts || "1",
      // Email notification settings
      mail_password_reset: settings.mail_password_reset || "1",
      mail_account_verification: settings.mail_account_verification || "1", 
      mail_notifications: settings.mail_notifications || "1",
      mail_test_mode: settings.mail_test_mode || "0",
      // Storage settings
      max_file_size: settings.max_file_size || "10",
      allowed_file_types: settings.allowed_file_types || "jpg,jpeg,png,gif,mp4,mov,pdf,doc,docx",
      image_quality: settings.image_quality || "80",
      storage_driver: settings.storage_driver || "local",
      aws_access_key_id: settings.aws_access_key_id || "",
      aws_secret_access_key: settings.aws_secret_access_key || "",
      aws_default_region: settings.aws_default_region || "us-east-1",
      aws_bucket: settings.aws_bucket || "",
      auto_resize_images: settings.auto_resize_images || "1",
      generate_thumbnails: settings.generate_thumbnails || "1",
      thumbnail_width: settings.thumbnail_width || "300",
      thumbnail_height: settings.thumbnail_height || "300",
      cache_driver: settings.cache_driver || "file",
      cache_ttl: settings.cache_ttl || "60",
      // Security settings
      two_factor_auth: settings.two_factor_auth || "0",
      session_timeout: settings.session_timeout || "30",
      max_login_attempts: settings.max_login_attempts || "5",
      lockout_duration: settings.lockout_duration || "15",
      security_notifications: settings.security_notifications || "1",
      password_min_length: settings.password_min_length || "6",
      password_require_uppercase: settings.password_require_uppercase || "0",
      password_require_numbers: settings.password_require_numbers || "0",
      password_require_special: settings.password_require_special || "0",
      privacy_mode: settings.privacy_mode || "0",
      data_retention_days: settings.data_retention_days || "365",
      gdpr_compliance: settings.gdpr_compliance || "0",
      data_export_enabled: settings.data_export_enabled || "1",
      account_deletion_enabled: settings.account_deletion_enabled || "1",
      audit_logs_enabled: settings.audit_logs_enabled || "0",
    }
    const formChanged = JSON.stringify(formData) !== JSON.stringify(initialFormData)
    setHasUnsavedChanges(formChanged)
  }, [formData, settings])

  // Professional breadcrumbs with icons
  const breadcrumbs = [
    { 
      label: t("Dashboard"), 
      href: "/user/dashboard",
      icon: (
        <svg className="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
          <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M3 12l2-2m0 0l7-7 7 7M5 10v10a1 1 0 001 1h3m10-11l2 2m-2-2v10a1 1 0 01-1 1h-3m-6 0a1 1 0 001-1v-4a1 1 0 011-1h2a1 1 0 011 1v4a1 1 0 001 1m-6 0h6" />
        </svg>
      )
    },
    { 
      label: t("Settings"),
      icon: (
        <svg className="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
          <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M10.325 4.317c.426-1.756 2.924-1.756 3.35 0a1.724 1.724 0 002.573 1.066c1.543-.94 3.31.826 2.37 2.37a1.724 1.724 0 001.065 2.572c1.756.426 1.756 2.924 0 3.35a1.724 1.724 0 00-1.066 2.573c.94 1.543-.826 3.31-2.37 2.37a1.724 1.724 0 00-2.572 1.065c-.426 1.756-2.924 1.756-3.35 0a1.724 1.724 0 00-2.573-1.066c-1.543.94-3.31-.826-2.37-2.37a1.724 1.724 0 00-1.065-2.572c-1.756-.426-1.756-2.924 0-3.35a1.724 1.724 0 001.066-2.573c-.94-1.543.826-3.31 2.37-2.37.996.608 2.296.07 2.572-1.065z" />
        </svg>
      )
    }
  ]

  const handleInputChange = (field, value) => {
    setFormData(prev => ({ ...prev, [field]: value }))
    setHasUnsavedChanges(true)
  }

  const handleThemeChange = (field, value) => {
    setThemeData(prev => ({ ...prev, [field]: value }))
    setHasUnsavedChanges(true)
  }

  const handleSubmit = () => {
    setIsSubmitting(true)
    router.post(route('user.setting.store'), formData, {
      preserveScroll: true,
      preserveState: false,
      onSuccess: () => {
        setHasUnsavedChanges(false)
        // Backend already sends success message, no need for frontend toast
      },
      onFinish: () => {
        setIsSubmitting(false)
      },
      onError: () => {
        setIsSubmitting(false)
        toast.error(t("Failed to save settings"))
      },
    })
  }


  const handleThemeSubmit = () => {
    setIsSubmitting(true)
    router.post(route('user.setting.store'), themeData, {
      preserveScroll: true,
      preserveState: false,
      onSuccess: () => {
        setHasUnsavedChanges(false)
        // Backend already sends success message, no need for frontend toast
      },
      onFinish: () => {
        setIsSubmitting(false)
      },
      onError: () => {
        setIsSubmitting(false)
        toast.error(t("Failed to save theme settings"))
      },
    })
  }

  const renderSectionContent = () => {
    // Dynamic theme colors for consistent styling
    const primaryColor = themeData.primary_color || "#667eea"
    const secondaryColor = themeData.secondary_color || "#764ba2"
    const buttonGradient = `linear-gradient(135deg, ${primaryColor} 0%, ${secondaryColor} 100%)`
    const buttonHoverGradient = `linear-gradient(135deg, ${primaryColor}dd 0%, ${secondaryColor}dd 100%)`
    
    
    switch (activeSection) {
      case "app-setting":
        return (
          <PremiumSettingsCard
            variant="glass"
            className="min-h-[600px]"
            title={t("Application Settings")}
            description={t("Configure and customize your application settings, preferences, and integrations")}
            icon={
              <svg className="w-5 h-5" fill="currentColor" viewBox="0 0 20 20">
                <path fillRule="evenodd" d="M11.49 3.17c-.38-1.56-2.6-1.56-2.98 0a1.532 1.532 0 01-2.286.948c-1.372-.836-2.942.734-2.106 2.106.54.886.061 2.042-.947 2.287-1.561.379-1.561 2.6 0 2.978a1.532 1.532 0 01.947 2.287c-.836 1.372.734 2.942 2.106 2.106a1.532 1.532 0 012.287.947c.379 1.561 2.6 1.561 2.978 0a1.533 1.533 0 012.287-.947c1.372.836 2.942-.734 2.106-2.106a1.533 1.533 0 01.947-2.287c1.561-.379 1.561-2.6 0-2.978a1.532 1.532 0 01-.947-2.287c.836-1.372-.734-2.942-2.106-2.106a1.532 1.532 0 01-2.287-.947zM10 13a3 3 0 100-6 3 3 0 000 6z" clipRule="evenodd" />
              </svg>
            }
          >
            <PremiumSettingsTabs
              activeTab={activeTab}
              onTabChange={setActiveTab}
              tabs={[
                { id: "basic-setting", label: t("General Settings") },
                { id: "theme-setting", label: t("Theme & Appearance") },
                { id: "logo-setting", label: t("Logo & Branding") },
                { id: "logging", label: t("System Logs") },
              ]}
            />
            
            <AnimatePresence mode="wait">
              <motion.div
                key={activeTab}
                initial={{ opacity: 0, x: 20 }}
                animate={{ opacity: 1, x: 0 }}
                exit={{ opacity: 0, x: -20 }}
                transition={{ duration: 0.2 }}
                className="mt-6"
              >
                {activeTab === "basic-setting" && (
                  <div className="space-y-6">
                    {/* Header */}
                    <div className="flex items-center justify-between mb-6">
                      <div>
                        <h3 className="text-lg font-semibold text-gray-900 dark:text-gray-100">{t("General Settings")}</h3>
                        <p className="text-sm text-gray-500 dark:text-gray-400 mt-1">
                          {t("Configure general application settings and preferences")}
                        </p>
                      </div>
                      <div className="flex space-x-3">
                        <Button
                          variant="outline"
                          onClick={() => window.location.reload()}
                          disabled={!hasUnsavedChanges}
                          className={`px-4 py-2.5 border border-gray-300 dark:border-zinc-600 text-gray-700 dark:text-gray-300 hover:bg-gray-50 dark:hover:bg-zinc-800 rounded-lg font-medium transition-colors ${!hasUnsavedChanges ? 'opacity-50 cursor-not-allowed' : ''}`}
                        >
                          {t("Cancel")}
                        </Button>
                        <Button
                          onClick={handleSubmit}
                          isLoading={isSubmitting}
                          className="bg-gradient-to-r from-primary to-secondary hover:from-primary/90 hover:to-secondary/90 text-white px-6 py-2.5 rounded-lg font-medium shadow-sm transition-all duration-200 flex items-center gap-2"
                        >
                          {!isSubmitting && (
                            <svg className="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                              <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M5 13l4 4L19 7" />
                            </svg>
                          )}
                          {isSubmitting ? t("Saving...") : t("Save Changes")}
                        </Button>
                      </div>
                    </div>

                    {/* Unsaved Changes Banner */}
                    {hasUnsavedChanges && (
                      <div className="bg-yellow-50 dark:bg-yellow-900/20 border border-yellow-200 dark:border-yellow-800 rounded-lg p-4 mb-6">
                        <div className="flex items-center">
                          <svg className="w-5 h-5 text-yellow-600 dark:text-yellow-400 mr-3" fill="currentColor" viewBox="0 0 20 20">
                            <path fillRule="evenodd" d="M18 10a8 8 0 11-16 0 8 8 0 0116 0zm-7 4a1 1 0 11-2 0 1 1 0 012 0zm-1-9a1 1 0 00-1 1v4a1 1 0 102 0V6a1 1 0 00-1-1z" clipRule="evenodd" />
                          </svg>
                          <div className="flex-1">
                            <p className="text-sm font-medium text-yellow-800 dark:text-yellow-200">
                              {t("You have unsaved changes")}
                            </p>
                            <p className="text-xs text-yellow-700 dark:text-yellow-300 mt-1">
                              {t("Don't forget to save your changes before leaving this page")}
                            </p>
                          </div>
                        </div>
                      </div>
                    )}

                    <SettingsSection title={t("Company Information")}>
                      <SettingsRow
                        label={t("Site Name")}
                        description={t("Your company or website name")}
                        required
                      >
                        <Input
                          value={formData.site_name}
                          onChange={(e) => handleInputChange("site_name", e.target.value)}
                          placeholder={t("Enter site name")}
                          className="max-w-md"
                        />
                      </SettingsRow>
                      
                      <SettingsRow
                        label={t("Contact Email")}
                        description={t("Primary contact email address")}
                        required
                      >
                        <Input
                          type="email"
                          value={formData.email}
                          onChange={(e) => handleInputChange("email", e.target.value)}
                          placeholder={t("email@example.com")}
                          className="max-w-md"
                        />
                      </SettingsRow>

                      <SettingsRow
                        label={t("Phone Number")}
                        description={t("Contact phone number")}
                      >
                        <div className="flex space-x-2 max-w-md">
                          <Input
                            value={formData.phone_code}
                            onChange={(e) => handleInputChange("phone_code", e.target.value)}
                            placeholder="+1"
                            className="w-20"
                          />
                          <Input
                            value={formData.phone}
                            onChange={(e) => handleInputChange("phone", e.target.value)}
                            placeholder={t("Phone number")}
                            className="flex-1"
                          />
                        </div>
                      </SettingsRow>
                    </SettingsSection>

                    <SettingsSection title={t("User Experience")}>
                      <SettingsRow
                        label={t("Enable Preloader")}
                        description={t("Show loading animation when navigating between pages")}
                      >
                        <div className="flex items-center space-x-3">
                          <Switch
                            checked={formData.preloader_enabled === "1"}
                            onCheckedChange={(checked) => 
                              handleInputChange("preloader_enabled", checked ? "1" : "0")
                            }
                            className="data-[state=checked]:bg-gradient-to-r data-[state=checked]:from-sky-400 data-[state=checked]:to-cyan-400"
                          />
                          <span className="text-sm text-gray-600 dark:text-gray-400">
                            {formData.preloader_enabled === "1" ? t("Enabled") : t("Disabled")}
                          </span>
                        </div>
                      </SettingsRow>
                    </SettingsSection>

                    <SettingsSection title={t("Regional Settings")}>
                      <SettingsRow
                        label={t("Time Zone")}
                        description={t("Your local time zone")}
                      >
                        <Select 
                          value={formData.time_zone || ""} 
                          onValueChange={(value) => handleInputChange("time_zone", value)}
                        >
                          <SelectTrigger className="max-w-md h-9 px-3 py-2 bg-gray-50 dark:bg-zinc-800 border-gray-200 dark:border-zinc-800 text-sm">
                            <SelectValue placeholder={t("Select time zone")} />
                          </SelectTrigger>
                          <SelectContent>
                            {settingConfig?.time_zone && Object.entries(settingConfig.time_zone).map(([key, value]) => (
                              <SelectItem key={key} value={key}>
                                {value}
                              </SelectItem>
                            ))}
                          </SelectContent>
                        </Select>
                      </SettingsRow>

                      <SettingsRow
                        label={t("Date Format")}
                        description={t("How dates are displayed")}
                      >
                        <Select 
                          value={formData.date_format || ""} 
                          onValueChange={(value) => handleInputChange("date_format", value)}
                        >
                          <SelectTrigger className="max-w-md h-9 px-3 py-2 bg-gray-50 dark:bg-zinc-800 border-gray-200 dark:border-zinc-800 text-sm">
                            <SelectValue placeholder={t("Select date format")} />
                          </SelectTrigger>
                          <SelectContent>
                            {settingConfig?.date_format?.map(format => (
                              <SelectItem key={format.format} value={format.format}>
                                {format.label}
                              </SelectItem>
                            ))}
                          </SelectContent>
                        </Select>
                      </SettingsRow>
                    </SettingsSection>
                  </div>
                )}
                
                {activeTab === "theme-setting" && (
                  <ThemeSetting
                    themeData={themeData}
                    onThemeChange={handleThemeChange}
                    onSubmit={handleThemeSubmit}
                    onReset={() => {
                      setThemeData({
                        primary_color: "#667eea",
                        secondary_color: "#764ba2",
                        primary_color_text: "#ffffff",
                        secondary_color_text: "#ffffff",
                        body_text_primary: "#1a202c",
                        body_text_secondary: "#718096",
                      })
                    }}
                    isSubmitting={isSubmitting}
                  />
                )}
                
                {activeTab === "logo-setting" && <LogoSetting />}
                {activeTab === "logging" && <Logging settings={settings} />}
              </motion.div>
            </AnimatePresence>
          </PremiumSettingsCard>
        )

      case "ai-configuration":
        return (
          <PremiumSettingsCard
            title={t("AI Configuration")}
            description={t("Configure AI modules and settings")}
            icon={
              <svg className="w-5 h-5" fill="currentColor" viewBox="0 0 20 20">
                <path d="M13 7H7v6h6V7z" />
                <path fillRule="evenodd" d="M7 2a1 1 0 012 0v1h2V2a1 1 0 112 0v1h2a2 2 0 012 2v2h1a1 1 0 110 2h-1v2h1a1 1 0 110 2h-1v2a2 2 0 01-2 2h-2v1a1 1 0 11-2 0v-1H9v1a1 1 0 11-2 0v-1H5a2 2 0 01-2-2v-2H2a1 1 0 110-2h1V9H2a1 1 0 010-2h1V5a2 2 0 012-2h2V2z" clipRule="evenodd" />
              </svg>
            }
            badge={{ text: "Pro", type: "pro" }}
          >
            <PremiumSettingsTabs
              activeTab={activeAiTab}
              onTabChange={setActiveAiTab}
              tabs={[
                { id: "text-ai", label: t("Text AI") },
                { id: "image-ai", label: t("Image AI") },
                { id: "video-ai", label: t("Video AI") },
              ]}
            />
            
            <AnimatePresence mode="wait">
              <motion.div
                key={activeAiTab}
                initial={{ opacity: 0, x: 20 }}
                animate={{ opacity: 1, x: 0 }}
                exit={{ opacity: 0, x: -20 }}
                transition={{ duration: 0.2 }}
                className="mt-6"
              >
                {activeAiTab === "text-ai" && (
                  <TextModule settingConfig={settingConfig} textAiSetting={aiSettings?.find(s => s.module === "text_ai")} />
                )}
                {activeAiTab === "image-ai" && (
                  <ImageModule settingConfig={settingConfig} imageAiSetting={aiSettings?.find(s => s.module === "image_ai")} />
                )}
                {activeAiTab === "video-ai" && (
                  <VideoModule settingConfig={settingConfig} videoAiSetting={aiSettings?.find(s => s.module === "video_ai")} />
                )}
              </motion.div>
            </AnimatePresence>
          </PremiumSettingsCard>
        )

      case "languages":
        return (
          <PremiumSettingsCard
            title={t("Language & Region")}
            description={t("Manage languages and regional settings")}
            icon={
              <svg className="w-5 h-5" fill="currentColor" viewBox="0 0 20 20">
                <path fillRule="evenodd" d="M7 2a1 1 0 011 1v1h3a1 1 0 110 2H9.578a18.87 18.87 0 01-1.724 4.78c.29.354.596.696.914 1.026a1 1 0 11-1.44 1.389c-.188-.196-.373-.396-.554-.6a19.098 19.098 0 01-3.107 3.567 1 1 0 01-1.334-1.49 17.087 17.087 0 003.13-3.733 18.992 18.992 0 01-1.487-2.494 1 1 0 111.79-.89c.234.47.489.928.764 1.372.417-.934.752-1.913.997-2.927H3a1 1 0 110-2h3V3a1 1 0 011-1zm6 6a1 1 0 01.894.553l2.991 5.982a.869.869 0 01.02.037l.99 1.98a1 1 0 11-1.79.895L15.383 16h-4.764l-.724 1.447a1 1 0 11-1.788-.894l.99-1.98.019-.038 2.99-5.982A1 1 0 0113 8zm-1.382 6h2.764L13 11.236 11.618 14z" clipRule="evenodd" />
              </svg>
            }
          >
            <div className="space-y-6">
              {/* Header */}
              <div className="flex items-center justify-between mb-6">
                <div>
                  <h3 className="text-lg font-semibold text-gray-900 dark:text-gray-100">{t("Language & Regional Settings")}</h3>
                  <p className="text-sm text-gray-500 dark:text-gray-400 mt-1">
                    {t("Manage application languages, regional preferences, and localization settings")}
                  </p>
                </div>
                <div className="flex gap-3">
                  <button
                    onClick={() => setIsLanguageModalOpen(true)}
                    className="inline-flex items-center px-4 py-2.5 bg-gradient-to-r from-primary to-secondary hover:from-primary/90 hover:to-secondary/90 text-white text-sm font-medium rounded-lg shadow-sm transition-all duration-200"
                  >
                    <svg className="w-4 h-4 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                      <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M12 4v16m8-8H4" />
                    </svg>
                    {t("Add Language")}
                  </button>
                </div>
              </div>

              <Languages
                languages={languages}
                isModalOpen={isLanguageModalOpen}
                setIsModalOpen={setIsLanguageModalOpen}
              />
            </div>
          </PremiumSettingsCard>
        )

      case "backup-setting":
        return (
          <BackupSettings 
            settings={settings} 
            backups={backups} 
            backupPath={backupPath} 
          />
        )

      case "automation":
        return (
          <PremiumSettingsCard
            title={t("Automation & Cron Jobs")}
            description={t("Manage automated tasks and scheduled processes")}
            icon={
              <svg className="w-5 h-5" fill="currentColor" viewBox="0 0 20 20">
                <path d="M13 6a3 3 0 11-6 0 3 3 0 016 0zM18 8a2 2 0 11-4 0 2 2 0 014 0zM14 15a4 4 0 00-8 0v3h8v-3z" />
              </svg>
            }
            badge={{ text: "Pro", type: "pro" }}
          >
            <AutomationSettings 
              cronJobs={cronJobs}
              systemStatus={systemStatus}
              cronUrls={[]}
            />
          </PremiumSettingsCard>
        )

      case "mail-setting":
        return (
          <PremiumSettingsCard
            title={t("Email Configuration")}
            description={t("Configure SMTP settings and email notifications")}
            icon={
              <svg className="w-5 h-5" fill="currentColor" viewBox="0 0 20 20">
                <path d="M2.003 5.884L10 9.882l7.997-3.998A2 2 0 0016 4H4a2 2 0 00-1.997 1.884z" />
                <path d="M18 8.118l-8 4-8-4V14a2 2 0 002 2h12a2 2 0 002-2V8.118z" />
              </svg>
            }
          >
            <div className="space-y-6">
              {/* Header */}
              <div className="flex items-center justify-between mb-6">
                <div>
                  <h3 className="text-lg font-semibold text-gray-900 dark:text-gray-100">{t("Email Configuration")}</h3>
                  <p className="text-sm text-gray-500 dark:text-gray-400 mt-1">
                    {t("Configure SMTP server settings and email notification preferences")}
                  </p>
                </div>
                <div className="flex space-x-3">
                  <Button
                    variant="outline"
                    onClick={() => window.location.reload()}
                    disabled={!hasUnsavedChanges}
                    className={`px-4 py-2.5 border border-gray-300 dark:border-zinc-600 text-gray-700 dark:text-gray-300 hover:bg-gray-50 dark:hover:bg-zinc-800 rounded-lg font-medium transition-colors ${!hasUnsavedChanges ? 'opacity-50 cursor-not-allowed' : ''}`}
                  >
                    {t("Cancel")}
                  </Button>
                  <Button
                    onClick={handleSubmit}
                    isLoading={isSubmitting}
                    className="bg-gradient-to-r from-primary to-secondary hover:from-primary/90 hover:to-secondary/90 text-white px-6 py-2.5 rounded-lg font-medium shadow-sm transition-all duration-200 flex items-center gap-2"
                  >
                    {!isSubmitting && (
                      <svg className="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M5 13l4 4L19 7" />
                      </svg>
                    )}
                    {isSubmitting ? t("Saving...") : t("Save Email Settings")}
                  </Button>
                </div>
              </div>

              <SettingsSection title={t("SMTP Configuration")}>
                <SettingsRow
                  label={t("SMTP Host")}
                  description={t("Your email server hostname")}
                  required
                >
                  <Input
                    value={formData.smtp_host || ""}
                    onChange={(e) => handleInputChange("smtp_host", e.target.value)}
                    placeholder={t("smtp.gmail.com")}
                    className="max-w-md"
                  />
                </SettingsRow>
                
                <SettingsRow
                  label={t("SMTP Port")}
                  description={t("Email server port (usually 587 or 465)")}
                  required
                >
                  <Input
                    type="number"
                    value={formData.smtp_port || ""}
                    onChange={(e) => handleInputChange("smtp_port", e.target.value)}
                    placeholder="587"
                    className="max-w-md"
                  />
                </SettingsRow>

                <SettingsRow
                  label={t("SMTP Username")}
                  description={t("Your email account username")}
                  required
                >
                  <Input
                    type="email"
                    value={formData.smtp_username || ""}
                    onChange={(e) => handleInputChange("smtp_username", e.target.value)}
                    placeholder={t("your-email@domain.com")}
                    className="max-w-md"
                  />
                </SettingsRow>

                <SettingsRow
                  label={t("SMTP Password")}
                  description={t("Your email account password")}
                  required
                >
                  <Input
                    type="password"
                    value={formData.smtp_password || ""}
                    onChange={(e) => handleInputChange("smtp_password", e.target.value)}
                    placeholder={t("Your password")}
                    className="max-w-md"
                  />
                </SettingsRow>

                <SettingsRow
                  label={t("Encryption")}
                  description={t("SMTP encryption method")}
                >
                  <Select 
                    value={formData.smtp_encryption || "tls"} 
                    onValueChange={(value) => handleInputChange("smtp_encryption", value)}
                  >
                    <SelectTrigger className="h-9 px-3 py-2 bg-gray-50 dark:bg-zinc-800 border-gray-200 dark:border-zinc-800 text-sm">
                      <SelectValue placeholder={t("Select encryption method")} />
                    </SelectTrigger>
                    <SelectContent>
                      <SelectItem value="tls">{t("TLS")}</SelectItem>
                      <SelectItem value="ssl">{t("SSL")}</SelectItem>
                      <SelectItem value="none">{t("None")}</SelectItem>
                    </SelectContent>
                  </Select>
                </SettingsRow>

                <SettingsRow
                  label={t("From Address")}
                  description={t("Default sender email address")}
                  required
                >
                  <Input
                    type="email"
                    value={formData.mail_from_address || ""}
                    onChange={(e) => handleInputChange("mail_from_address", e.target.value)}
                    placeholder={t("noreply@yourdomain.com")}
                    className="max-w-md"
                  />
                </SettingsRow>

                <SettingsRow
                  label={t("From Name")}
                  description={t("Default sender name")}
                >
                  <Input
                    value={formData.mail_from_name || ""}
                    onChange={(e) => handleInputChange("mail_from_name", e.target.value)}
                    placeholder={t("Your App Name")}
                    className="max-w-md"
                  />
                </SettingsRow>
              </SettingsSection>

              <SettingsSection title={t("Email Notifications")}>
                <ToggleCard
                  title={t("Password Reset Emails")}
                  description={t("Send password reset emails to users")}
                  enabled={formData.mail_password_reset === "1"}
                  onToggle={() => {
                    const newValue = formData.mail_password_reset === "1" ? "0" : "1"
                    handleInputChange("mail_password_reset", newValue)
                    // Auto-save this setting
                    setTimeout(() => {
                      router.post(route('user.setting.store'), {
                        mail_password_reset: newValue
                      })
                    }, 300)
                  }}
                />

                <ToggleCard
                  title={t("Account Verification Emails")}
                  description={t("Send verification emails for new accounts")}
                  enabled={formData.mail_account_verification === "1"}
                  onToggle={() => {
                    const newValue = formData.mail_account_verification === "1" ? "0" : "1"
                    handleInputChange("mail_account_verification", newValue)
                    // Auto-save this setting
                    setTimeout(() => {
                      router.post(route('user.setting.store'), {
                        mail_account_verification: newValue
                      })
                    }, 300)
                  }}
                />

                <ToggleCard
                  title={t("General Notifications")}
                  description={t("Send general notification emails")}
                  enabled={formData.mail_notifications === "1"}
                  onToggle={() => {
                    const newValue = formData.mail_notifications === "1" ? "0" : "1"
                    handleInputChange("mail_notifications", newValue)
                    // Auto-save this setting
                    setTimeout(() => {
                      router.post(route('user.setting.store'), {
                        mail_notifications: newValue
                      })
                    }, 300)
                  }}
                />

                <ToggleCard
                  title={t("Test Mode")}
                  description={t("Enable test mode for email sending")}
                  enabled={formData.mail_test_mode === "1"}
                  onToggle={() => {
                    const newValue = formData.mail_test_mode === "1" ? "0" : "1"
                    handleInputChange("mail_test_mode", newValue)
                    // Auto-save this setting
                    setTimeout(() => {
                      router.post(route('user.setting.store'), {
                        mail_test_mode: newValue
                      })
                    }, 300)
                  }}
                />
              </SettingsSection>

              <SettingsSection title={t("Test Email")}>
                <SettingsRow
                  label={t("Send Test Email")}
                  description={t("Send a test email to verify your configuration")}
                >
                  <div className="flex items-center space-x-2 max-w-md">
                    <Input
                      type="email"
                      placeholder={t("test@example.com")}
                      className="flex-1"
                      id="test-email-input"
                    />
                    <Button 
                      variant="outline"
                      onClick={() => {
                        const email = document.getElementById('test-email-input')?.value;
                        if (email) {
                          // Add test email functionality here
                          router.post(route('user.setting.test-email'), {
                            ...formData,
                            test_email: email
                          });
                        }
                      }}
                    >
                      {t("Send Test")}
                    </Button>
                  </div>
                </SettingsRow>
              </SettingsSection>
            </div>
          </PremiumSettingsCard>
        )

      case "security":
        return (
          <PremiumSettingsCard
            title={t("Security & Privacy")}
            description={t("Configure security settings and privacy options")}
            icon={
              <svg className="w-5 h-5" fill="currentColor" viewBox="0 0 20 20">
                <path fillRule="evenodd" d="M2.166 4.999A11.954 11.954 0 0010 1.944 11.954 11.954 0 0017.834 5c.11.65.166 1.32.166 2.001 0 5.225-3.34 9.67-8 11.317C5.34 16.67 2 12.225 2 7c0-.682.057-1.35.166-2.001zm11.541 3.708a1 1 0 00-1.414-1.414L9 10.586 7.707 9.293a1 1 0 00-1.414 1.414l2 2a1 1 0 001.414 0l4-4z" clipRule="evenodd" />
              </svg>
            }
          >
            <div className="space-y-6">
              {/* Header */}
              <div className="flex items-center justify-between mb-6">
                <div>
                  <h3 className="text-lg font-semibold text-gray-900 dark:text-gray-100">{t("Security & Privacy Settings")}</h3>
                  <p className="text-sm text-gray-500 dark:text-gray-400 mt-1">
                    {t("Configure authentication, password policies, and privacy options")}
                  </p>
                </div>
                <div className="flex space-x-3">
                  <Button
                    variant="outline"
                    onClick={() => window.location.reload()}
                    disabled={!hasUnsavedChanges}
                    className={`px-4 py-2.5 border border-gray-300 dark:border-zinc-600 text-gray-700 dark:text-gray-300 hover:bg-gray-50 dark:hover:bg-zinc-800 rounded-lg font-medium transition-colors ${!hasUnsavedChanges ? 'opacity-50 cursor-not-allowed' : ''}`}
                  >
                    {t("Cancel")}
                  </Button>
                  <Button
                    onClick={handleSubmit}
                    isLoading={isSubmitting}
                    className="bg-gradient-to-r from-primary to-secondary hover:from-primary/90 hover:to-secondary/90 text-white px-6 py-2.5 rounded-lg font-medium shadow-sm transition-all duration-200 flex items-center gap-2"
                  >
                    {!isSubmitting && (
                      <svg className="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M5 13l4 4L19 7" />
                      </svg>
                    )}
                    {isSubmitting ? t("Saving...") : t("Save Security Settings")}
                  </Button>
                </div>
              </div>

              <SettingsSection title={t("Authentication Security")}>
                <div className="space-y-4">
                   
                  <ToggleCard
                    title={t("Enforce 2FA for All Users")}
                    description={t("Require all users to enable two-factor authentication")}
                    enabled={formData.two_factor_auth === "1"}
                    onToggle={() => {
                      const newValue = formData.two_factor_auth === "1" ? "0" : "1"
                      handleInputChange("two_factor_auth", newValue)
                      // Auto-save this setting
                      setTimeout(() => {
                        router.post(route('user.setting.store'), {
                          two_factor_auth: newValue
                        }, {
                          preserveScroll: true,
                          preserveState: true
                        })
                      }, 500)
                    }}
                  />
                </div>

                <SettingsRow
                  label={t("Session Timeout")}
                  description={t("Auto-logout users after inactivity (minutes)")}
                >
                  <Select 
                    value={formData.session_timeout || "30"} 
                    onValueChange={(value) => handleInputChange("session_timeout", value)}
                  >
                    <SelectTrigger className="w-full h-9 px-3 py-2 bg-gray-50 dark:bg-zinc-800 border-gray-200 dark:border-zinc-800 text-sm">
                      <SelectValue placeholder={t("Select timeout duration")} />
                    </SelectTrigger>
                    <SelectContent>
                      <SelectItem value="15">{t("15 minutes")}</SelectItem>
                      <SelectItem value="30">{t("30 minutes")}</SelectItem>
                      <SelectItem value="60">{t("1 hour")}</SelectItem>
                      <SelectItem value="120">{t("2 hours")}</SelectItem>
                      <SelectItem value="480">{t("8 hours")}</SelectItem>
                    </SelectContent>
                  </Select>
                </SettingsRow>

                <SettingsRow
                  label={t("Max Login Attempts")}
                  description={t("Maximum failed login attempts before lockout")}
                >
                  <Input
                    type="number"
                    value={formData.max_login_attempts || "5"}
                    onChange={(e) => handleInputChange("max_login_attempts", e.target.value)}
                    placeholder="5"
                    className="max-w-md"
                    min="1"
                    max="10"
                  />
                </SettingsRow>

                <SettingsRow
                  label={t("Lockout Duration")}
                  description={t("Account lockout duration (minutes)")}
                >
                  <Input
                    type="number"
                    value={formData.lockout_duration || "15"}
                    onChange={(e) => handleInputChange("lockout_duration", e.target.value)}
                    placeholder="15"
                    className="max-w-md"
                    min="5"
                    max="1440"
                  />
                </SettingsRow>

                <ToggleCard
                  title={t("Security Notifications")}
                  description={t("Send notifications for security events")}
                  enabled={formData.security_notifications === "1"}
                  onToggle={() => {
                    const newValue = formData.security_notifications === "1" ? "0" : "1"
                    handleInputChange("security_notifications", newValue)
                    // Auto-save this setting
                    setTimeout(() => {
                      router.post(route('user.setting.store'), {
                        security_notifications: newValue
                      }, {
                        preserveScroll: true,
                        preserveState: true
                      })
                    }, 500)
                  }}
                />
              </SettingsSection>

              <SettingsSection title={t("Password Security")}>
                <SettingsRow
                  label={t("Minimum Password Length")}
                  description={t("Minimum number of characters required")}
                >
                  <Input
                    type="number"
                    value={formData.password_min_length || "6"}
                    onChange={(e) => handleInputChange("password_min_length", e.target.value)}
                    placeholder="6"
                    className="max-w-md"
                    min="4"
                    max="50"
                  />
                </SettingsRow>

                <ToggleCard
                  title={t("Require Uppercase Letters")}
                  description={t("Password must contain uppercase letters")}
                  enabled={formData.password_require_uppercase === "1"}
                  onToggle={() => {
                    const newValue = formData.password_require_uppercase === "1" ? "0" : "1"
                    handleInputChange("password_require_uppercase", newValue)
                    setTimeout(() => {
                      router.post(route('user.setting.store'), {
                        password_require_uppercase: newValue
                      }, {
                        preserveScroll: true,
                        preserveState: true
                      })
                    }, 500)
                  }}
                />

                <ToggleCard
                  title={t("Require Numbers")}
                  description={t("Password must contain numbers")}
                  enabled={formData.password_require_numbers === "1"}
                  onToggle={() => {
                    const newValue = formData.password_require_numbers === "1" ? "0" : "1"
                    handleInputChange("password_require_numbers", newValue)
                    setTimeout(() => {
                      router.post(route('user.setting.store'), {
                        password_require_numbers: newValue
                      }, {
                        preserveScroll: true,
                        preserveState: true
                      })
                    }, 500)
                  }}
                />

                <ToggleCard
                  title={t("Require Special Characters")}
                  description={t("Password must contain special characters")}
                  enabled={formData.password_require_special === "1"}
                  onToggle={() => {
                    const newValue = formData.password_require_special === "1" ? "0" : "1"
                    handleInputChange("password_require_special", newValue)
                    setTimeout(() => {
                      router.post(route('user.setting.store'), {
                        password_require_special: newValue
                      }, {
                        preserveScroll: true,
                        preserveState: true
                      })
                    }, 500)
                  }}
                />
              </SettingsSection>

              <SettingsSection title={t("Privacy & Data")}>
                <ToggleCard
                  title={t("Privacy Mode")}
                  description={t("Hide sensitive information in logs")}
                  enabled={formData.privacy_mode === "1"}
                  onToggle={() => {
                    const newValue = formData.privacy_mode === "1" ? "0" : "1"
                    handleInputChange("privacy_mode", newValue)
                    setTimeout(() => {
                      router.post(route('user.setting.store'), {
                        privacy_mode: newValue
                      }, {
                        preserveScroll: true,
                        preserveState: true
                      })
                    }, 500)
                  }}
                />

                <SettingsRow
                  label={t("Data Retention Days")}
                  description={t("Days to retain user data after deletion")}
                >
                  <Input
                    type="number"
                    value={formData.data_retention_days || "365"}
                    onChange={(e) => handleInputChange("data_retention_days", e.target.value)}
                    placeholder="365"
                    className="max-w-md"
                    min="30"
                    max="3650"
                  />
                </SettingsRow>

                <ToggleCard
                  title={t("GDPR Compliance")}
                  description={t("Enable GDPR compliance features")}
                  enabled={formData.gdpr_compliance === "1"}
                  onToggle={() => {
                    const newValue = formData.gdpr_compliance === "1" ? "0" : "1"
                    handleInputChange("gdpr_compliance", newValue)
                    setTimeout(() => {
                      router.post(route('user.setting.store'), {
                        gdpr_compliance: newValue
                      }, {
                        preserveScroll: true,
                        preserveState: true
                      })
                    }, 500)
                  }}
                />

                <ToggleCard
                  title={t("Data Export")}
                  description={t("Allow users to export their data")}
                  enabled={formData.data_export_enabled === "1"}
                  onToggle={() => {
                    const newValue = formData.data_export_enabled === "1" ? "0" : "1"
                    handleInputChange("data_export_enabled", newValue)
                    setTimeout(() => {
                      router.post(route('user.setting.store'), {
                        data_export_enabled: newValue
                      }, {
                        preserveScroll: true,
                        preserveState: true
                      })
                    }, 500)
                  }}
                />

                <ToggleCard
                  title={t("Account Deletion")}
                  description={t("Allow users to delete their accounts")}
                  enabled={formData.account_deletion_enabled === "1"}
                  onToggle={() => {
                    const newValue = formData.account_deletion_enabled === "1" ? "0" : "1"
                    handleInputChange("account_deletion_enabled", newValue)
                    setTimeout(() => {
                      router.post(route('user.setting.store'), {
                        account_deletion_enabled: newValue
                      }, {
                        preserveScroll: true,
                        preserveState: true
                      })
                    }, 500)
                  }}
                />

                <ToggleCard
                  title={t("Audit Logs")}
                  description={t("Enable detailed audit logging")}
                  enabled={formData.audit_logs_enabled === "1"}
                  onToggle={() => {
                    const newValue = formData.audit_logs_enabled === "1" ? "0" : "1"
                    handleInputChange("audit_logs_enabled", newValue)
                    setTimeout(() => {
                      router.post(route('user.setting.store'), {
                        audit_logs_enabled: newValue
                      }, {
                        preserveScroll: true,
                        preserveState: true
                      })
                    }, 500)
                  }}
                />
              </SettingsSection>
            </div>
          </PremiumSettingsCard>
        )

      case "storage-setting":
        return (
          <PremiumSettingsCard
            title={t("Storage & Media")}
            description={t("Manage file storage and media settings")}
            icon={
              <svg className="w-5 h-5" fill="currentColor" viewBox="0 0 20 20">
                <path d="M3 4a1 1 0 011-1h12a1 1 0 011 1v2a1 1 0 01-1 1H4a1 1 0 01-1-1V4zM3 10a1 1 0 011-1h6a1 1 0 011 1v6a1 1 0 01-1 1H4a1 1 0 01-1-1v-6zM14 9a1 1 0 00-1 1v6a1 1 0 001 1h2a1 1 0 001-1v-6a1 1 0 00-1-1h-2z" />
              </svg>
            }
          >
            <div className="space-y-6">
              {/* Header */}
              <div className="flex items-center justify-between mb-6">
                <div>
                  <h3 className="text-lg font-semibold text-gray-900 dark:text-gray-100">{t("Storage & Media Settings")}</h3>
                  <p className="text-sm text-gray-500 dark:text-gray-400 mt-1">
                    {t("Configure file storage, media processing, and cache settings")}
                  </p>
                </div>
                <Button
                  onClick={handleSubmit}
                  isLoading={isSubmitting}
                  className="bg-gradient-to-r from-primary to-secondary hover:from-primary/90 hover:to-secondary/90 text-white px-6 py-2.5 rounded-lg font-medium shadow-sm transition-all duration-200 flex items-center gap-2"
                >
                  {!isSubmitting && (
                    <svg className="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                      <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M5 13l4 4L19 7" />
                    </svg>
                  )}
                  {isSubmitting ? t("Saving...") : t("Save Storage Settings")}
                </Button>
              </div>

              <SettingsSection title={t("File Upload Settings")}>
                <SettingsRow label={t("Maximum File Size")} description={t("Maximum size for file uploads (MB)")}>
                  <Input
                    type="number"
                    value={formData.max_file_size || "10"}
                    onChange={(e) => handleInputChange("max_file_size", e.target.value)}
                    min="1"
                    max="100"
                  />
                </SettingsRow>
                <SettingsRow label={t("Allowed File Types")} description={t("Comma-separated list of allowed file extensions")}>
                  <Input
                    value={formData.allowed_file_types || "jpg,jpeg,png,gif,mp4,mov,pdf,doc,docx"}
                    onChange={(e) => handleInputChange("allowed_file_types", e.target.value)}
                    placeholder="jpg,jpeg,png,gif,mp4,mov,pdf"
                  />
                </SettingsRow>
                <SettingsRow label={t("Image Quality")} description={t("JPEG compression quality (1-100)")}>
                  <Input
                    type="number"
                    value={formData.image_quality || "80"}
                    onChange={(e) => handleInputChange("image_quality", e.target.value)}
                    min="1"
                    max="100"
                  />
                </SettingsRow>
              </SettingsSection>

              <SettingsSection title={t("Storage Configuration")}>
                <SettingsRow label={t("Storage Driver")} description={t("Select storage driver for file uploads")}>
                  <Select 
                    value={formData.storage_driver || "local"} 
                    onValueChange={(value) => handleInputChange("storage_driver", value)}
                  >
                    <SelectTrigger className="w-full h-9 px-3 py-2 bg-gray-50 dark:bg-zinc-800 border-gray-200 dark:border-zinc-800 text-sm">
                      <SelectValue placeholder={t("Select storage driver")} />
                    </SelectTrigger>
                    <SelectContent>
                      <SelectItem value="local">{t("Local Storage")}</SelectItem>
                      <SelectItem value="s3">{t("Amazon S3")}</SelectItem>
                      <SelectItem value="digitalocean">{t("DigitalOcean Spaces")}</SelectItem>
                      <SelectItem value="google">{t("Google Cloud Storage")}</SelectItem>
                    </SelectContent>
                  </Select>
                </SettingsRow>
                
                {formData.storage_driver === "s3" && (
                  <>
                    <SettingsRow label={t("AWS Access Key")} required>
                      <Input
                        value={formData.aws_access_key_id || ""}
                        onChange={(e) => handleInputChange("aws_access_key_id", e.target.value)}
                        placeholder={t("Your AWS Access Key")}
                      />
                    </SettingsRow>
                    <SettingsRow label={t("AWS Secret Key")} required>
                      <Input
                        type="password"
                        value={formData.aws_secret_access_key || ""}
                        onChange={(e) => handleInputChange("aws_secret_access_key", e.target.value)}
                        placeholder={t("Your AWS Secret Key")}
                      />
                    </SettingsRow>
                    <SettingsRow label={t("AWS Region")} required>
                      <Input
                        value={formData.aws_default_region || "us-east-1"}
                        onChange={(e) => handleInputChange("aws_default_region", e.target.value)}
                        placeholder="us-east-1"
                      />
                    </SettingsRow>
                    <SettingsRow label={t("S3 Bucket Name")} required>
                      <Input
                        value={formData.aws_bucket || ""}
                        onChange={(e) => handleInputChange("aws_bucket", e.target.value)}
                        placeholder={t("your-bucket-name")}
                      />
                    </SettingsRow>
                  </>
                )}
              </SettingsSection>

              <SettingsSection title={t("Media Processing")}>
                <SettingsRow label={t("Auto-resize Images")}>
                  <Switch
                    checked={formData.auto_resize_images === "1" || formData.auto_resize_images === true}
                    onCheckedChange={(checked) => handleInputChange("auto_resize_images", checked ? "1" : "0")}
                  />
                </SettingsRow>
                <SettingsRow label={t("Generate Thumbnails")}>
                  <Switch
                    checked={formData.generate_thumbnails === "1" || formData.generate_thumbnails === true}
                    onCheckedChange={(checked) => handleInputChange("generate_thumbnails", checked ? "1" : "0")}
                  />
                </SettingsRow>
                <SettingsRow label={t("Thumbnail Size")} description={t("Thumbnail dimensions in pixels")}>
                  <div className="flex space-x-2">
                    <Input
                      type="number"
                      value={formData.thumbnail_width || "300"}
                      onChange={(e) => handleInputChange("thumbnail_width", e.target.value)}
                      placeholder="300"
                    />
                    <span className="flex items-center text-gray-500">×</span>
                    <Input
                      type="number"
                      value={formData.thumbnail_height || "300"}
                      onChange={(e) => handleInputChange("thumbnail_height", e.target.value)}
                      placeholder="300"
                    />
                  </div>
                </SettingsRow>
              </SettingsSection>

              <SettingsSection title={t("Cache Settings")}>
                <SettingsRow label={t("Cache Driver")} description={t("Select cache driver for application")}>
                  <Select 
                    value={formData.cache_driver || "file"} 
                    onValueChange={(value) => handleInputChange("cache_driver", value)}
                  >
                    <SelectTrigger className="w-full h-9 px-3 py-2 bg-gray-50 dark:bg-zinc-800 border-gray-200 dark:border-zinc-800 text-sm">
                      <SelectValue placeholder={t("Select cache driver")} />
                    </SelectTrigger>
                    <SelectContent>
                      <SelectItem value="file">{t("File Cache")}</SelectItem>
                      <SelectItem value="redis">{t("Redis")}</SelectItem>
                      <SelectItem value="memcached">{t("Memcached")}</SelectItem>
                      <SelectItem value="database">{t("Database")}</SelectItem>
                    </SelectContent>
                  </Select>
                </SettingsRow>
                <SettingsRow label={t("Cache TTL")} description={t("Cache time-to-live in minutes")}>
                  <Input
                    type="number"
                    value={formData.cache_ttl || "60"}
                    onChange={(e) => handleInputChange("cache_ttl", e.target.value)}
                    min="1"
                    placeholder="60"
                  />
                </SettingsRow>
              </SettingsSection>
            </div>
          </PremiumSettingsCard>
        )

      default:
        return (
          <PremiumSettingsCard
            title={t("Coming Soon")}
            description={t("This feature is under development")}
            variant="gradient"
          >
            <div className="text-center py-12">
              <div className="w-24 h-24 bg-gray-100 dark:bg-zinc-800 rounded-full flex items-center justify-center mx-auto mb-4">
                <svg className="w-12 h-12 text-gray-400" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                  <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={1.5} d="M19 11H5m14 0a2 2 0 012 2v6a2 2 0 01-2 2H5a2 2 0 01-2-2v-6a2 2 0 012-2m14 0V9a2 2 0 00-2-2M5 11V9a2 2 0 012-2m0 0V5a2 2 0 012-2h6a2 2 0 012 2v2M7 7h10" />
                </svg>
              </div>
              <h3 className="text-lg font-semibold text-gray-900 dark:text-white mb-2">
                {t("Feature Under Development")}
              </h3>
              <p className="text-sm text-gray-500 dark:text-gray-400 max-w-md mx-auto">
                {t("We're working hard to bring you this feature. Stay tuned for updates!")}
              </p>
            </div>
          </PremiumSettingsCard>
        )
    }
  }

  return (
    <AppLayout breadcrumbs={breadcrumbs}>
      <div className="min-h-screen bg-gray-50 dark:bg-zinc-950">
        {/* Premium Header */}
        <div className="p-6">
          <div className="px-0 py-0">
            <div className="flex items-center justify-between">
              <div>
                <div  className="flex items-center gap-2 sm:gap-3">
                  <Settings className="h-5 w-5 sm:h-6 sm:w-6 text-primary" />
                  <h1 className="text-xl sm:text-2xl font-bold text-gray-900 dark:text-white">
                  {t("Settings")}
                </h1>
                </div>
                <div>
                  <p className="mt-2 text-gray-500 dark:text-gray-400">
                    {t("Manage your application preferences and configurations")}
                  </p>
                </div>
              </div>
              
              {/* Quick Actions */}
              <div className="flex items-center space-x-3">
                <Button
                  variant="outline"
                  className="flex items-center space-x-2"
                  onClick={() => router.get(route('user.profile.settings'))}
                >
                  <svg
                    className="w-4 h-4 text-gray-700 dark:text-gray-200"
                    fill="none"
                    stroke="currentColor"
                    viewBox="0 0 24 24"
                  >
                    <path
                      strokeLinecap="round"
                      strokeLinejoin="round"
                      strokeWidth={2}
                      d="M16 7a4 4 0 11-8 0 4 4 0 018 0zM12 14a7 7 0 00-7 7h14a7 7 0 00-7-7z"
                    />
                  </svg>
                  <span className="text-gray-700 dark:text-gray-200">{t("Profile")}</span>
                </Button>

              </div>
            </div>
          </div>
        </div>

        {/* Main Content */}
        <div className="p-6">
          <div className="w-full mx-auto">
            <div className="flex gap-6 lg:flex-row flex-col">
              {/* Premium Sidebar */}
              <PremiumSettingsSidebar
                activeSection={activeSection}
                onSectionChange={setActiveSection}
                themeData={themeData}
                lastBackupTime={lastBackupTime}
              />

              {/* Content Area */}
              <div className="flex-1">
                {renderSectionContent()}
              </div>
            </div>
          </div>
        </div>
      </div>
    </AppLayout>
  )
}