<?php

namespace App\Http\Services\Account\Instagram;

use App\Enums\ConnectionType;
use App\Traits\AccountManager;
use App\Enums\PostType;
use App\Models\Platform;
use App\Models\SocialAccount;
use App\Models\SocialPost;
use Exception;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\Http;
use App\Models\Core\File;

class Account
{


    use AccountManager;



    const BASE_URL = 'https://www.facebook.com';
    const API_URL = 'https://graph.facebook.com';


    /**
     * Summary of getScopes
     * @return array
     */
    public static function getScopes(string $type = 'auth'): array
    {


        switch ($type) {

            case 'auth':
                return [
                    'ads_management',
                    'business_management',
                    'instagram_basic',
                    'instagram_content_publish',
                    'pages_read_engagement'
                ];

            default:

                return [
                    'pages_read_engagement'
                ];
        }

    }




    /**
     * Summary of getApiUrl
     * @param string $endpoint
     * @param array $params
     * @param mixed $configuration
     * @param bool $isBaseUrl
     * @return mixed
     */
    public static function getApiUrl(string $endpoint, array $params = [], mixed $configuration, bool $isBaseUrl = false): mixed
    {


        $apiUrl = $isBaseUrl ? self::BASE_URL : $configuration->graph_api_url;

        if (str_starts_with($endpoint, '/'))
            $endpoint = substr($endpoint, 1);

        $v = $configuration->app_version;

        $versionedUrlWithEndpoint = $apiUrl . '/' . ($v ? ($v . '/') : '') . $endpoint;

        if (count($params))
            $versionedUrlWithEndpoint .= '?' . http_build_query($params);


        return $versionedUrlWithEndpoint;


    }





    /**
     * Summary of getAccessToken
     * @param string $code
     * @param \App\Models\Platform $platform
     * @return \Illuminate\Http\Client\Response
     */
    public static function getAccessToken(string $code, Platform $platform)
    {

        $configuration = $platform->configuration;
        $apiUrl = self::getApiUrl('/oauth/access_token', [
            'code' => $code,
            'client_id' => $configuration->client_id,
            'client_secret' => $configuration->client_secret,
            'redirect_uri' => url('/account/instagram/callback?medium=' . $platform->slug),
        ], $configuration);

        return Http::post($apiUrl);
    }





    /**
     * Summary of refreshAccessToken
     * @param \App\Models\Platform $platform
     * @param string $token
     * @return \Illuminate\Http\Client\Response
     */
    public static function refreshAccessToken(Platform $platform, string $token): \Illuminate\Http\Client\Response
    {
        $configuration = $platform->configuration;


        $apiUrl = self::getApiUrl('/oauth/access_token', [
            'client_id' => $configuration->client_id,
            'client_secret' => $configuration->client_secret,
            'grant_type' => 'fb_exchange_token',
            'fb_exchange_token' => $token,
        ], $configuration);

        return Http::post($apiUrl);
    }




    /**
     * Summary of authRedirect
     * @param \App\Models\Platform $platform
     * @return mixed
     */
    public static function authRedirect(Platform $platform)
    {


        $scopes = collect(self::getScopes())->join(',');
        $configuration = $platform->configuration;


        return self::getApiUrl('dialog/oauth', [
            'response_type' => 'code',
            'client_id' => $configuration->client_id,
            'redirect_uri' => url('/account/instagram/callback?medium=' . $platform->slug),
            'scope' => $scopes,
        ], $configuration, true);


    }





    /**
     * Summary of getPagesInfo
     * @param array $fields
     * @param \App\Models\Platform $platform
     * @param string $token
     * @return \Illuminate\Http\Client\Response
     */
    public static function getAccounts(array $fields = ['connected_instagram_account,name,access_token'], Platform $platform, string $token): \Illuminate\Http\Client\Response
    {

        $configuration = $platform->configuration;

        $apiUrl = self::getApiUrl('/me/accounts', [
            'access_token' => $token,
            'fields' => collect($fields)->join(',')
        ], $configuration);

        return Http::get($apiUrl);
    }






    /**
     * Summary of getInstagramAccountInfo
     * @param string $accountId
     * @param array $fields
     * @param \App\Models\Platform $platform
     * @param string $token
     * @return \Illuminate\Http\Client\Response
     */
    public function getInstagramAccountInfo(string $accountId, array $fields = null, Platform $platform, string $token): \Illuminate\Http\Client\Response
    {

        $configuration = $platform->configuration;

        $redirect_uri = self::getApiUrl(endpoint: '/' . $accountId, params: [], configuration: $configuration);

        return Http::withToken($token)->get($redirect_uri, [
            'fields' => collect($fields)->join(','),
        ]);
    }


    /**
     * Summary of saveFbAccount
     * @param mixed $pages
     * @param string $guard
     * @param \App\Models\Platform $platform
     * @param string $is_official
     * @param int|string $dbId
     * @return void
     */
    public static function saveIgAccount(
        mixed $pages,
        string $guard,
        Platform $platform,
        string $account_type,
        string $is_official,
        string $token,
        int|string $dbId = null
    ) {





        $instagram = new self();



        foreach ($pages as $page) {




            if (isset($page['connected_instagram_account']) && isset($page['connected_instagram_account']['id'])) {


                $pageId = $page['connected_instagram_account']['id'];




                try {



                    $account = $instagram->getInstagramAccountInfo(
                        $pageId,
                        ['id,name,username,profile_picture_url'],
                        $platform,
                        $token
                    )->throw()
                        ->json();

                    $accountInfo = [

                        'id' => $account['id'],
                        'account_id' => $account['id'],
                        'name' => Arr::get($account, 'username', null),
                        'avatar' => Arr::get($account, 'profile_picture_url', null),
                        'email' => Arr::get($account, 'email', null),
                        'token' => Arr::get($account, 'access_token', $token),

                        'access_token_expire_at' => now()->addMonths(2),
                        'refresh_token' => Arr::get($account, 'access_token', $token),
                        'refresh_token_expire_at' => now()->addMonths(2),

                    ];

                    $response = $instagram->saveAccount($guard, $platform, $accountInfo, $account_type, $is_official, $dbId);


                } catch (\Exception $ex) {

                }




            }


        }
    }






    public function accountDetails(SocialAccount $account): array
    {


        try {

            $baseApi = $account->platform->configuration->graph_api_url;
            $apiVersion = $account->platform->configuration->app_version;
            $api = $baseApi . "/" . $apiVersion;
            $token = $account->token;
            $userId = $account->account_id;
            $apiUrl = $api . "/" . $userId . "/media";
            $fields = 'id,caption,media_type,media_url,thumbnail_url,permalink,timestamp,like_count,comments_count';

            $params = [
                'fields' => $fields,
                'access_token' => $token,
            ];

            $response = Http::get($apiUrl, $params);
            $apiResponse = $response->json();

            if (isset($apiResponse['error'])) {

                $this->disConnectAccount($account);
                return [
                    'status' => false,
                    'message' => $apiResponse['error']['message']
                ];
            }

            $apiResponse = $this->formatResponse($apiResponse);

            return [
                'status' => true,
                'response' => $apiResponse,
            ];


        } catch (\Exception $ex) {

            return [
                'status' => false,
                'message' => strip_tags($ex->getMessage())
            ];
        }

    }

    public function formatResponse(array $response): array
    {


        $responseData = Arr::get($response, 'data', []);

        if (count($responseData) > 0) {

            $formattedData = [];
            foreach ($responseData as $key => $value) {
                $formattedData[] = [
                    'status_type' => Arr::get($value, 'media_type', null),
                    'full_picture' => Arr::get($value, 'media_url', null),
                    'link' => Arr::get($value, 'permalink', null),
                    'created_time' => Arr::get($value, 'timestamp', null),
                    'reactions' => [
                        'summary' => [
                            'total_count' => Arr::get($value, 'like_count', 0), // Instagram likes
                        ],
                    ],
                    'comments' => [
                        'summary' => [
                            'total_count' => Arr::get($value, 'comments_count', 0), // Instagram comments
                        ],
                    ]
                ];
            }
            $response['data'] = $formattedData;
        }

        return $response;


    }


    public function send(SocialPost $post): array
    {
        $account = $post->account;
        if ($account->is_official == ConnectionType::OFFICIAL->value) {
            return $this->official($post);
        }

        return [
            'status' => false,
            'response' => 'Only official connections are supported for Instagram',
            'url' => null
        ];
    }


    public function official(SocialPost $post): array
    {

        try {

            $account = $post->account;
            $platform = @$post->account->platform;

            $accountConnection = $this->accountDetails($account);

            $isConnected = Arr::get($accountConnection, 'status', false);
            $message = translate("Text and url not supported in insta feed");
            $status = false;
            $token = $account->account_information->token;

            if ($post->files && $post->files->count() > 0) {

                $message = translate("Text and url not supported in insta feed");


                if ($post->files && $post->files->count() > 0) {

                    $message = translate("Posted Successfully");
                    $status = true;



                    #POST IN FEED
                    if ($post->post_type == PostType::FEED->value) {
                        $response = $this->postFeed($post, $platform);
                    }
                    #POST IN REELS
                    elseif ($post->post_type == PostType::REELS->value) {
                        $response = $this->postReel($post, $platform);
                    }
                    #POST IN STORY
                    elseif ($post->post_type == PostType::STORY->value) {
                        $response = $this->postStory($post, $platform);
                    }

                    $url = Arr::get($response, 'url');
                    $message = Arr::get($response, 'message', $message);
                    $status = Arr::get($response, 'status', $status);
                    $postId = Arr::get($response, 'post_id');

                }

            }


        } catch (\Exception $ex) {
            $status = false;
            $message = strip_tags($ex->getMessage());
        }

        return [
            'status' => @$url ? true : false,
            'response' => @$message,
            'url' => @$url,
            'post_id' => @$postId ?? null, // Numeric media ID for analytics
        ];

    }



    /**
     * Summary of postFeed
     * @param \App\Models\SocialPost $post
     * @param mixed $ig
     * @return array
     */
    public function postFeed(SocialPost $post, Platform $platform): array
    {

        $account = $post->account;

        $token = $account->account_information->token;


        $media_ids = [];

        if ($post->files->count() == 0) {

            return [

                'status' => false,
                'message' => translate('Instagram doesn\'t support  only text posting'),
            ];

        }




        if ($post->files->count() == 0) {

            return [

                'status' => false,
                'message' => translate('Instagram doesn\'t support  only text posting'),
            ];

        }




        if ($post->files->count() > 10) {

            return [

                'status' => false,
                'message' => translate('Instagram doesn\'t support more than 10 media at a time'),
            ];

        }




        if ($post->files->count() > 1) {

            #MULTI FILE POST


            $accountId = $account->account_id;

            $configuration = $platform->configuration;

            $apiUrl = self::getApiUrl($accountId . '/media', [], $configuration);

            foreach ($post->files as $file) {

                $fileURL = imageURL($file, "gallery");

                $upload_params['is_carousel_item'] = true;
                $upload_params['caption'] = $post->content ?? "feed";


                if (!isValidVideoUrl($fileURL)) {

                    $upload_params['media_type'] = 'IMAGE';
                    $upload_params['image_url'] = $fileURL;

                } else {

                    $upload_params['media_type'] = 'VIDEO';
                    $upload_params['video_url'] = $fileURL;

                }


                $upload_response = Http::withToken($token)
                    ->asForm()
                    ->acceptJson()
                    ->post($apiUrl, $upload_params)
                    ->throw();


                $media_ids[] = @$upload_response->json('id');

            }

            $upload_params = [
                'media_type' => 'CAROUSEL',
                'children' => $media_ids,
                'caption' => $post->content ?? "feed"
            ];


            $publishCarouselResponse = Http::withToken($token)
                ->retry(3, 3000)
                ->post($apiUrl, $upload_params);




            $uploadResponse = $this->publishContainer(
                $accountId,
                $publishCarouselResponse->json('id'),
                $platform,
                $token
            );



            if (@$uploadResponse["id"]) {
                $mediaId = $uploadResponse["id"];
                $shortcode = $this->getPost($mediaId, $token, $platform);
                $url = "https://www.instagram.com/p/" . $shortcode;
            }

        } else {

            #SINGLE FILE POST

            $file = $post->files->first();

            $response = $this->publishSingleFile($file, $account, $platform, $post->content ?? "feed");

            if (@$response["id"]) {
                $mediaId = $response["id"];
                $shortcode = $this->getPost($mediaId, $token, $platform);
                $url = "https://www.instagram.com/p/" . $shortcode;
            }

        }

        return [
            'url' => @$url,
            'status' => @$url ? true : false,
            'message' => @$url ? translate('Posted successfully') : translate('Failed to post'),
            'post_id' => @$mediaId ?? null, // Return numeric media ID for analytics
        ];

    }



    /**
     * Summary of getPost
     * @param mixed $postId
     * @param mixed $token
     * @param mixed $platform
     */
    public function getPost($postId, $token, $platform)
    {

        $configuration = $platform->configuration;

        $response = Http::withToken($token)
            ->get(self::getApiUrl($postId . "?fields=shortcode", [], $configuration))->throw();



        return @$response->json('shortcode');



    }






    /**
     * Summary of publishSingleFile
     * @param \App\Models\Core\File $file
     * @param \App\Models\SocialAccount $account
     * @param \App\Models\Platform $platform
     * @param mixed $caption
     * @return array
     */
    public function publishSingleFile(File $file, SocialAccount $account, Platform $platform, $caption): array
    {


        $id = $account->account_id;

        $configuration = $platform->configuration;
        $token = $account->token;

        $fileURL = imageURL($file, "gallery");



        if (!isValidVideoUrl($fileURL)) {

            $postData = ['image_url' => imageURL($file, "gallery"), 'caption' => $caption];
        } else {

            $postData = [
                'media_type' => "REELS",
                'video_url' => $fileURL,
                'share_to_feed' => true,
                'caption' => $caption
            ];

        }


        $apiUrl = self::getApiUrl("$id/media", [], $configuration);

        $response = Http::withToken($token)
            ->retry(3, sleepMilliseconds: 3000)
            ->post($apiUrl, $postData)->throw();


        $mediaId = $response->json('id');

        $isUploaded = $this->checkUploadStatus(
            mediaId: $mediaId,
            delayInSeconds: 3,
            maxAttempts: 10,
            platform: $platform,
            token: $token
        );



        if (!$isUploaded['is_ready']) {

            return [
                'status' => false,
            ];
        }


        $uploadResponse = $this->publishContainer($id, $mediaId, $platform, $token);


        return ['id' => $uploadResponse->json('id')];





    }





    /**
     * Summary of publishContainer
     * @param string $igId
     * @param string|int $mediaId
     * @param \App\Models\Platform $platform
     * @param string $token
     * @return \Illuminate\Http\Client\Response
     */
    protected function publishContainer(string $igId, string|int $mediaId, Platform $platform, string $token)
    {


        $configuration = $platform->configuration;

        $apiUrl = self::getApiUrl($igId . '/media_publish', [], $configuration);

        return Http::retry(3, 3000)
            ->withToken($token)
            ->post($apiUrl, [
                'creation_id' => (int) $mediaId,
            ]);
    }








    private function checkUploadStatus(
        string $mediaId,
        int $delayInSeconds = 3,
        int $maxAttempts = 10,
        Platform $platform,
        string $token
    ): array {
        $status = false;
        $attempted = 0;
        $isFinished = false;

        $configuration = $platform->configuration;

        while (!$isFinished && $attempted < $maxAttempts) {

            $videoStatus = Http::withToken($token)
                ->baseUrl(self::API_URL)
                ->retry(1, 3000)
                ->get(self::getApiUrl($mediaId, ['fields' => 'status_code,status'], $configuration))->throw();


            $status = $videoStatus->json('status_code');
            $isFinished = in_array(strtolower($status), ['finished', 'ok', 'completed', 'ready']);

            if ($isFinished) {
                break;
            }

            $isError = in_array(strtolower($status), ['error', 'failed']);
            if ($isError) {
                break;
            }

            $attempted++;
            sleep($delayInSeconds);
        }

        return [
            'is_ready' => $isFinished,
            'status_code' => $status,
            'status' => $videoStatus->json('status'),
        ];
    }





    /**
     * Summary of postReel
     * @param \App\Models\SocialPost $post
     * @param \App\Models\Platform $platform
     * @return array
     */
    public function postReel(SocialPost $post, Platform $platform): array
    {

        $account = $post->account;
        $token = $account->account_information->token;
        $file = $post->files->first();
        $fileURL = imageURL($file, "gallery");


        $id = $account->account_id;

        $configuration = $platform->configuration;
        $token = $account->token;


        if (isValidVideoUrl($fileURL)) {


            $postData = [
                'media_type' => "REELS",
                'video_url' => $fileURL,
                'share_to_feed' => true,
                'caption' => $post->content ?? "feed"
            ];



            $apiUrl = self::getApiUrl("$id/media", [], $configuration);

            $response = Http::withToken($token)
                ->retry(3, sleepMilliseconds: 3000)
                ->post($apiUrl, $postData)->throw();


            $mediaId = $response->json('id');

            $isUploaded = $this->checkUploadStatus(
                mediaId: $mediaId,
                delayInSeconds: 3,
                maxAttempts: 10,
                platform: $platform,
                token: $token
            );



            if (!$isUploaded['is_ready']) {

                return [
                    'status' => false,
                ];
            }


            $uploadResponse = $this->publishContainer($id, $mediaId, $platform, $token);


            if (@$uploadResponse["id"]) {
                $publishedMediaId = $uploadResponse["id"];
                $shortcode = $this->getPost($publishedMediaId, $token, $platform);
                $url = "https://www.instagram.com/p/" . $shortcode;

                return [
                    'url' => @$url,
                    'status' => true,
                    'message' => translate('Posted successfully'),
                    'post_id' => $publishedMediaId, // Return numeric media ID for analytics
                ];
            }
        }

        return [
            "status" => false,
            "message" => translate("Instagram reels doesnot support uploading images")
        ];


    }






    /**
     * Summary of postStory
     * @param \App\Models\SocialPost $post
     * @param \App\Models\Platform $platform
     * @return array
     */
    public function postStory(SocialPost $post, Platform $platform): array
    {

        $account = $post->account;
        $token = $account->account_information->token;

        $configuration = $platform->configuration;
        $file = $post->files->first();
        $fileURL = imageURL($file, "gallery");

        $id = $account->account_id;

        $postData['caption'] = $post->content ?? "feed";
        $postData['media_type'] = "STORIES";

        if (isValidVideoUrl($fileURL)) {
            $postData['video_url'] = $fileURL;
        } else {
            $postData['image_url'] = $fileURL;
        }



        $apiUrl = self::getApiUrl("$id/media", [], $configuration);

        $response = Http::withToken($token)
            ->retry(3, sleepMilliseconds: 3000)
            ->post($apiUrl, $postData)->throw();


        $mediaId = $response->json('id');

        $isUploaded = $this->checkUploadStatus(
            mediaId: $mediaId,
            delayInSeconds: 3,
            maxAttempts: 10,
            platform: $platform,
            token: $token
        );


        if (!$isUploaded['is_ready']) {

            return [
                'status' => false,
                'message' => translate('Failed to post')
            ];
        }

        $uploadResponse = $this->publishContainer($id, $mediaId, $platform, $token);

        $publishedMediaId = null;
        if (@$uploadResponse["id"]) {
            $publishedMediaId = $uploadResponse["id"];
            $shortcode = $this->getPost($publishedMediaId, $token, $platform);
            $url = "https://www.instagram.com/p/" . $shortcode;
        }

        return [
            'url' => @$url,
            'status' => @$url ? true : false,
            'message' => @$url ? translate('Posted successfully') : translate('Failed to post'),
            'post_id' => $publishedMediaId, // Return numeric media ID for analytics
        ];
    }

    /**
     * Get account-level insights (followers count)
     * https://developers.facebook.com/docs/instagram-api/reference/ig-user
     *
     * @param SocialAccount $account
     * @return array
     */
    public function getAccountInsights(SocialAccount $account): array
    {
        try {
            $token = $account->token ?? $account->access_token ?? null;

            if (!$token || !$account->account_id) {
                return [
                    'status' => false,
                    'message' => 'Missing access token or account ID',
                    'data' => [],
                ];
            }

            $configuration = $account->platform->configuration ?? null;

            // Get Instagram Business/Creator account info with followers_count
            $apiUrl = self::getApiUrl($account->account_id, [
                'fields' => 'followers_count,media_count,username',
                'access_token' => $token,
            ], $configuration);

            $response = Http::get($apiUrl);
            $data = $response->json();

            if ($response->failed() || isset($data['error'])) {
                $errorMsg = $data['error']['message'] ?? 'API request failed';
                return [
                    'status' => false,
                    'message' => $errorMsg,
                    'data' => [],
                ];
            }

            return [
                'status' => true,
                'message' => 'Account insights fetched successfully',
                'data' => [
                    'followers' => $data['followers_count'] ?? 0,
                ],
            ];

        } catch (\Throwable $e) {
            return [
                'status' => false,
                'message' => 'Error fetching account insights: ' . $e->getMessage(),
                'data' => [],
            ];
        }
    }

    /**
     * Get metrics/insights for an Instagram post
     * https://developers.facebook.com/docs/instagram-api/reference/ig-media/insights
     *
     * Note: As of March 2025, Meta deprecated 'impressions' in favor of 'views'
     * Also deprecated: plays, initial_plays for Reels
     *
     * @param SocialPost $post
     * @param SocialAccount $account
     * @return array
     */
    public function getInsight(SocialPost $post, SocialAccount $account): array
    {
        try {
            $token = $account->token ?? $account->access_token ?? null;

            if (!$token || !$post->platform_post_id) {
                return [
                    'status' => false,
                    'message' => 'Missing access token or media ID',
                    'metrics' => [],
                ];
            }

            $platform = $account->platform;
            $configuration = $platform->configuration;

            // First get basic media info (likes, comments are public)
            $mediaUrl = self::getApiUrl($post->platform_post_id, [
                'fields' => 'like_count,comments_count,media_type',
                'access_token' => $token,
            ], $configuration);

            $mediaResponse = Http::get($mediaUrl);
            $mediaData = $mediaResponse->json();

            if ($mediaResponse->failed() || isset($mediaData['error'])) {
                $errorMsg = $mediaData['error']['message'] ?? 'Media API request failed';
                return [
                    'status' => false,
                    'message' => $errorMsg,
                    'metrics' => [],
                ];
            }

            $likes = $mediaData['like_count'] ?? 0;
            $comments = $mediaData['comments_count'] ?? 0;
            $mediaType = $mediaData['media_type'] ?? 'IMAGE';

            // Determine which insights metrics to request based on media type
            // As of March 2025: 'impressions' deprecated -> use 'views'
            // 'plays' deprecated for Reels -> use 'views'
            $metricsToRequest = ['reach', 'saved'];

            // Try 'views' first (new metric), fallback to 'impressions' for older API versions
            if (in_array($mediaType, ['VIDEO', 'REELS'])) {
                $metricsToRequest[] = 'views';
            } else {
                // For images/carousels, try views first
                $metricsToRequest[] = 'views';
            }

            // Fetch insights with new metrics
            $insightsUrl = self::getApiUrl("{$post->platform_post_id}/insights", [
                'metric' => implode(',', $metricsToRequest),
                'access_token' => $token,
            ], $configuration);

            $insightsResponse = Http::get($insightsUrl);
            $insightsData = $insightsResponse->json();

            $views = 0;
            $reach = 0;
            $saved = 0;

            // If views metric failed (older API version), try with impressions
            if (isset($insightsData['error']) && str_contains($insightsData['error']['message'] ?? '', 'views')) {
                $fallbackMetrics = ['reach', 'saved', 'impressions'];
                if (in_array($mediaType, ['VIDEO', 'REELS'])) {
                    $fallbackMetrics[] = 'plays';
                }

                $insightsUrl = self::getApiUrl("{$post->platform_post_id}/insights", [
                    'metric' => implode(',', $fallbackMetrics),
                    'access_token' => $token,
                ], $configuration);

                $insightsResponse = Http::get($insightsUrl);
                $insightsData = $insightsResponse->json();
            }

            $insights = $insightsData['data'] ?? [];

            // Parse insights into metrics
            foreach ($insights as $insight) {
                $name = $insight['name'] ?? '';
                $value = $insight['values'][0]['value'] ?? 0;

                switch ($name) {
                    case 'views':
                        $views = $value;
                        break;
                    case 'impressions':
                        // Fallback for older API versions
                        if ($views == 0) {
                            $views = $value;
                        }
                        break;
                    case 'reach':
                        $reach = $value;
                        break;
                    case 'saved':
                        $saved = $value;
                        break;
                    case 'plays':
                        // Fallback for video plays (deprecated)
                        if ($views == 0) {
                            $views = $value;
                        }
                        break;
                }
            }

            // Calculate engagements
            $engagements = $likes + $comments + $saved;

            // Use reach as fallback for views/impressions
            $impressionsValue = $views > 0 ? $views : $reach;

            $metrics = [
                'impressions' => $impressionsValue,
                'engagements' => $engagements,
                'reactions' => $likes,
                'comments' => $comments,
                'shares' => 0, // Instagram doesn't expose shares via API
                'likes' => $likes,
                'reach' => $reach > 0 ? $reach : $impressionsValue,
            ];

            return [
                'status' => true,
                'message' => 'Metrics fetched successfully',
                'metrics' => $metrics,
            ];

        } catch (\Throwable $e) {
            return [
                'status' => false,
                'message' => 'Error fetching Instagram metrics: ' . $e->getMessage(),
                'metrics' => [],
            ];
        }
    }
}
