<?php

namespace Database\Seeders;

use App\Enums\StatusEnum;
use App\Models\User;
use App\Models\Role;
use App\Models\Workspace;
use Illuminate\Database\Seeder;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Str;

class DemoUserSeeder extends Seeder
{
    /**
     * Run the database seeds for demo users.
     */
    public function run(): void
    {
        // Create admin/owner user
        $owner = User::updateOrCreate(
            ['email' => 'admin@postuno.com'],
            [
                'name' => 'Admin User',
                'username' => 'admin',
                'email' => 'admin@postuno.com',
                'phone' => '+12345678900',
                'phone_code' => '999',
                'country_code' => 'US9',
                'password' => Hash::make('password123'),
                'owner' => StatusEnum::true->status(),
                'status' => StatusEnum::true->status(),
                'email_verified_at' => now(),
                'last_login' => now(),
            ]
        );

        // Create workspace for owner
        $ownerWorkspace = Workspace::updateOrCreate(
            ['user_id' => $owner->id, 'name' => 'Admin Workspace'],
            [
                'email' => 'admin@postuno.com',
                'status' => StatusEnum::true->status(),
                'is_default' => StatusEnum::true->status(),
                'is_current' => StatusEnum::true->status(),
            ]
        );

        $owner->update(['current_workspace_id' => $ownerWorkspace->id]);

        // Get or create roles
        $managerRole = Role::firstOrCreate(
            ['name' => 'Manager'],
            [
                'uid' => Str::uuid(),
                'created_by' => $owner->id,
                'permissions' => json_encode([
                    'dashboard.view',
                    'posts.create',
                    'posts.edit',
                    'posts.delete',
                    'accounts.view',
                    'accounts.create',
                    'templates.view',
                    'templates.create',
                    'analytics.view',
                ]),
                'status' => StatusEnum::true->status(),
            ]
        );

        $editorRole = Role::firstOrCreate(
            ['name' => 'Editor'],
            [
                'uid' => Str::uuid(),
                'created_by' => $owner->id,
                'permissions' => json_encode([
                    'dashboard.view',
                    'posts.create',
                    'posts.edit',
                    'accounts.view',
                    'templates.view',
                    'templates.create',
                ]),
                'status' => StatusEnum::true->status(),
            ]
        );

        $viewerRole = Role::firstOrCreate(
            ['name' => 'Viewer'],
            [
                'uid' => Str::uuid(),
                'created_by' => $owner->id,
                'permissions' => json_encode([
                    'dashboard.view',
                    'posts.view',
                    'accounts.view',
                    'templates.view',
                    'analytics.view',
                ]),
                'status' => StatusEnum::true->status(),
            ]
        );

        // Create demo users with different roles
        $users = [
            [
                'name' => 'John Manager',
                'username' => 'john_manager',
                'email' => 'john@postuno.com',
                'role_id' => $managerRole->id,
                'phone' => '+12345678901',
                'phone_code' => '123',
                'country_code' => 'US1',
            ],
            [
                'name' => 'Sarah Editor',
                'username' => 'sarah_editor',
                'email' => 'sarah@postuno.com',
                'role_id' => $editorRole->id,
                'phone' => '+12345678902',
                'phone_code' => '456',
                'country_code' => 'CA1',
            ],
            [
                'name' => 'Mike Content',
                'username' => 'mike_content',
                'email' => 'mike@postuno.com',
                'role_id' => $editorRole->id,
                'phone' => '+12345678903',
                'phone_code' => '789',
                'country_code' => 'GB1',
            ],
            [
                'name' => 'Emma Viewer',
                'username' => 'emma_viewer',
                'email' => 'emma@postuno.com',
                'role_id' => $viewerRole->id,
                'phone' => '+12345678904',
                'phone_code' => '321',
                'country_code' => 'AU1',
            ],
            [
                'name' => 'David Analytics',
                'username' => 'david_analytics',
                'email' => 'david@postuno.com',
                'role_id' => $viewerRole->id,
                'phone' => '+12345678905',
                'phone_code' => '654',
                'country_code' => 'DE1',
            ],
        ];

        foreach ($users as $userData) {
            $user = User::updateOrCreate(
                ['email' => $userData['email']],
                [
                    'name' => $userData['name'],
                    'username' => $userData['username'],
                    'email' => $userData['email'],
                    'phone' => $userData['phone'],
                    'phone_code' => $userData['phone_code'],
                    'country_code' => $userData['country_code'],
                    'password' => Hash::make('password123'),
                    'owner' => StatusEnum::false->status(),
                    'status' => StatusEnum::true->status(),
                    'role_id' => $userData['role_id'],
                    'email_verified_at' => now(),
                    'last_login' => now()->subDays(rand(1, 30)),
                ]
            );

            // Create workspace for each user
            $workspace = Workspace::updateOrCreate(
                ['user_id' => $user->id, 'name' => $userData['name'] . ' Workspace'],
                [
                    'email' => $userData['email'],
                    'status' => StatusEnum::true->status(),
                    'is_default' => StatusEnum::true->status(),
                    'is_current' => StatusEnum::true->status(),
                ]
            );

            $user->update(['current_workspace_id' => $workspace->id]);
        }

        $this->command->info('Demo users seeded successfully!');
    }
}