"use client"

import { motion } from "framer-motion"
import { cn } from "@/Lib/utils"
import useTranslate from "@/hooks/useTranslate"

// Professional icon components
const icons = {
  "app-setting": (
    <svg className="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
      <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={1.5} d="M10.325 4.317c.426-1.756 2.924-1.756 3.35 0a1.724 1.724 0 002.573 1.066c1.543-.94 3.31.826 2.37 2.37a1.724 1.724 0 001.065 2.572c1.756.426 1.756 2.924 0 3.35a1.724 1.724 0 00-1.066 2.573c.94 1.543-.826 3.31-2.37 2.37a1.724 1.724 0 00-2.572 1.065c-.426 1.756-2.924 1.756-3.35 0a1.724 1.724 0 00-2.573-1.066c-1.543.94-3.31-.826-2.37-2.37a1.724 1.724 0 00-1.065-2.572c-1.756-.426-1.756-2.924 0-3.35a1.724 1.724 0 001.066-2.573c-.94-1.543.826-3.31 2.37-2.37.996.608 2.296.07 2.572-1.065z" />
      <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={1.5} d="M15 12a3 3 0 11-6 0 3 3 0 016 0z" />
    </svg>
  ),
  "ai-configuration": (
    <svg className="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
      <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={1.5} d="M9.75 17L9 20l-1 1h8l-1-1-.75-3M3 13h18M5 17h14a2 2 0 002-2V5a2 2 0 00-2-2H5a2 2 0 00-2 2v10a2 2 0 002 2z" />
    </svg>
  ),
  "languages": (
    <svg className="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
      <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={1.5} d="M3 5h12M9 3v2m1.048 9.5A18.022 18.022 0 016.412 9m6.088 9h7M11 21l5-10 5 10M12.751 5C11.783 10.77 8.07 15.61 3 18.129" />
    </svg>
  ),
  "storage-setting": (
    <svg className="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
      <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={1.5} d="M4 7v10c0 2.21 3.582 4 8 4s8-1.79 8-4V7M4 7c0 2.21 3.582 4 8 4s8-1.79 8-4M4 7c0-2.21 3.582-4 8-4s8 1.79 8 4m0 5c0 2.21-3.582 4-8 4s-8-1.79-8-4" />
    </svg>
  ),
  "mail-setting": (
    <svg className="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
      <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={1.5} d="M3 8l7.89 5.26a2 2 0 002.22 0L21 8M5 19h14a2 2 0 002-2V7a2 2 0 00-2-2H5a2 2 0 00-2 2v10a2 2 0 002 2z" />
    </svg>
  ),
  "security": (
    <svg className="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
      <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={1.5} d="M9 12l2 2 4-4m5.618-4.016A11.955 11.955 0 0112 2.944a11.955 11.955 0 01-8.618 3.04A12.02 12.02 0 003 9c0 5.591 3.824 10.29 9 11.622 5.176-1.332 9-6.03 9-11.622 0-1.042-.133-2.052-.382-3.016z" />
    </svg>
  ),
  "integrations": (
    <svg className="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
      <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={1.5} d="M11 4a2 2 0 114 0v1a1 1 0 001 1h3a1 1 0 011 1v3a1 1 0 01-1 1h-1a2 2 0 100 4h1a1 1 0 011 1v3a1 1 0 01-1 1h-3a1 1 0 01-1-1v-1a2 2 0 10-4 0v1a1 1 0 01-1 1H7a1 1 0 01-1-1v-3a1 1 0 00-1-1H4a2 2 0 110-4h1a1 1 0 001-1V7a1 1 0 011-1h3a1 1 0 001-1V4z" />
    </svg>
  ),
  "notifications": (
    <svg className="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
      <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={1.5} d="M15 17h5l-1.405-1.405A2.032 2.032 0 0118 14.158V11a6.002 6.002 0 00-4-5.659V5a2 2 0 10-4 0v.341C7.67 6.165 6 8.388 6 11v3.159c0 .538-.214 1.055-.595 1.436L4 17h5m6 0v1a3 3 0 11-6 0v-1m6 0H9" />
    </svg>
  ),
  "automation": (
    <svg className="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
      <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={1.5} d="M13 10V3L4 14h7v7l9-11h-7z" />
    </svg>
  )
}

export default function PremiumSettingsSidebar({ activeSection, onSectionChange, themeData, lastBackupTime }) {
  const t = useTranslate()
  
  // Function to format time difference
  const formatTimeAgo = (timestamp) => {
    if (!timestamp) return t("Never")
    
    const now = new Date()
    const backupDate = new Date(timestamp)
    const diffInMinutes = Math.floor((now - backupDate) / (1000 * 60))
    
    if (diffInMinutes < 1) return t("Just now")
    if (diffInMinutes < 60) return diffInMinutes === 1 ? t("1 minute ago") : t(`${diffInMinutes} minutes ago`)
    
    const diffInHours = Math.floor(diffInMinutes / 60)
    if (diffInHours < 24) return diffInHours === 1 ? t("1 hour ago") : t(`${diffInHours} hours ago`)
    
    const diffInDays = Math.floor(diffInHours / 24)
    if (diffInDays < 30) return diffInDays === 1 ? t("1 day ago") : t(`${diffInDays} days ago`)
    
    const diffInMonths = Math.floor(diffInDays / 30)
    return diffInMonths === 1 ? t("1 month ago") : t(`${diffInMonths} months ago`)
  }
  
  // Use dynamic theme colors or fallback to defaults
  const primaryColor = themeData?.primary_color || "#667eea"
  const secondaryColor = themeData?.secondary_color || "#764ba2"

  const sidebarSections = [
    {
      title: t("General"),
      items: [
        { id: "app-setting", label: t("Application Settings"), icon: icons["app-setting"], badge: null },
        { id: "ai-configuration", label: t("AI Configuration"), icon: icons["ai-configuration"], badge: null },
      ]
    },
    {
      title: t("Customization"),
      items: [
        { id: "languages", label: t("Language & Region"), icon: icons["languages"], badge: null },
        { id: "storage-setting", label: t("Storage & Media"), icon: icons["storage-setting"], badge: null },
      ]
    },
    {
      title: t("System"),
      items: [
        { id: "mail-setting", label: t("Email Configuration"), icon: icons["mail-setting"], badge: null },
        { id: "security", label: t("Security & Privacy"), icon: icons["security"], badge: null },
        { id: "automation", label: t("Automation & Cron Jobs"), icon: icons["automation"], badge: null },
        { id: "backup-setting", label: t("Backup & Maintenance"), icon: icons["integrations"], badge: null },
      ]
    }
  ]

  return (
    <div className="lg:w-72 w-full">
      <div className="bg-white dark:bg-zinc-900 rounded-xl shadow-sm border border-gray-100 dark:border-zinc-800 overflow-hidden">
        {/* Sidebar Header */}
        <div className="px-6 py-5 border-b border-gray-100 dark:border-zinc-800">
          <h3 className="text-lg font-semibold text-gray-900 dark:text-white">
            {t("Settings")}
          </h3>
          <p className="text-sm text-gray-500 dark:text-gray-400 mt-1">
            {t("Manage your preferences")}
          </p>
        </div>

        {/* Sidebar Navigation */}
        <div className="px-3 py-4">
          {sidebarSections.map((section, sectionIdx) => (
            <div key={sectionIdx} className={sectionIdx > 0 ? "mt-6" : ""}>
              {/* Section Title */}
              <h4 className="px-3 mb-2 text-xs font-semibold text-gray-400 dark:text-gray-500 uppercase tracking-wider">
                {section.title}
              </h4>

              {/* Section Items */}
              <div className="space-y-1">
                {section.items.map((item) => (
                  <motion.button
                    key={item.id}
                    onClick={() => onSectionChange(item.id)}
                    whileHover={{ x: 2 }}
                    whileTap={{ scale: 0.98 }}
                    className={cn(
                      "w-full px-3 py-2.5 rounded-lg flex items-center justify-between group transition-all duration-200",
                      activeSection === item.id
                        ? "text-white shadow-md"
                        : "hover:bg-gray-50 dark:hover:bg-zinc-800 text-gray-700 dark:text-gray-300"
                    )}
                    style={
                      activeSection === item.id
                        ? {
                            background: `linear-gradient(135deg, ${primaryColor} 0%, ${secondaryColor} 100%)`
                          }
                        : {}
                    }
                  >
                    <div className="flex items-center space-x-3">
                      <span className={cn(
                        "transition-colors",
                        activeSection === item.id
                          ? "text-white"
                          : "text-gray-400 dark:text-gray-500 group-hover:text-gray-600 dark:group-hover:text-gray-400"
                      )}>
                        {item.icon}
                      </span>
                      <span className="text-sm font-medium">
                        {item.label}
                      </span>
                    </div>

                    {/* Badge */}
                    {item.badge && (
                      <span className={cn(
                        "px-2 py-0.5 text-xs font-semibold rounded-full",
                        item.badge === "Pro" 
                          ? "bg-gradient-to-r from-yellow-400 to-orange-500 text-white"
                          : item.badge === "New"
                          ? "bg-green-100 text-green-700 dark:bg-green-900/30 dark:text-green-400"
                          : activeSection === item.id
                          ? "bg-white/20 text-white"
                          : "bg-gray-100 text-gray-600 dark:bg-zinc-700 dark:text-gray-300"
                      )}>
                        {item.badge}
                      </span>
                    )}
                  </motion.button>
                ))}
              </div>
            </div>
          ))}
        </div>

        {/* System Status Section */}
        <div className="p-4 mt-4 border-t border-gray-100 dark:border-zinc-800">
          <div 
            className="rounded-lg p-4"
            style={{
              background: `linear-gradient(135deg, ${primaryColor} 0%, ${secondaryColor} 100%)`
            }}
          >
            <div className="flex items-center space-x-3 mb-3">
              <div className="w-10 h-10 bg-white/20 backdrop-blur rounded-lg flex items-center justify-center">
                <svg className="w-6 h-6 text-white" fill="currentColor" viewBox="0 0 20 20">
                  <path fillRule="evenodd" d="M10 18a8 8 0 100-16 8 8 0 000 16zm3.707-9.293a1 1 0 00-1.414-1.414L9 10.586 7.707 9.293a1 1 0 00-1.414 1.414l2 2a1 1 0 001.414 0l4-4z" clipRule="evenodd" />
                </svg>
              </div>
              <div>
                <h5 className="text-white font-semibold text-sm">{t("System Status")}</h5>
                <p className="text-green-100 text-xs">{t("All systems operational")}</p>
              </div>
            </div>
            <div className="flex items-center justify-between text-sm text-white/90">
              <span>{t("Last backup")}</span>
              <span>{formatTimeAgo(lastBackupTime)}</span>
            </div>
          </div>
        </div>
      </div>
    </div>
  )
}