import { ThumbsUp, MessageCircle, Share, MoreHorizontal } from "lucide-react"

// Utility functions for hashtag detection and highlighting
const extractHashtags = (text) => {
  const hashtagRegex = /#[a-zA-Z0-9_]+/g
  return text.match(hashtagRegex) || []
}

const highlightContent = (text) => {
  const hashtagRegex = /#[a-zA-Z0-9_]+/g
  const urlRegex = /(https?:\/\/[^\s]+)/g

  // Split by both hashtags and URLs, keeping the delimiters
  const parts = text.split(/(#[a-zA-Z0-9_]+|https?:\/\/[^\s]+)/g)

  return parts
    .map((part, index) => {
      if (!part) return null

      if (hashtagRegex.test(part)) {
        return (
          <span
            key={index}
            className="bg-yellow-200 dark:bg-yellow-600 text-yellow-800 dark:text-yellow-200 px-1 rounded"
          >
            {part}
          </span>
        )
      }

      if (urlRegex.test(part)) {
        return (
          <span key={index} className="text-blue-600 dark:text-blue-400">
            {part}
          </span>
        )
      }

      return part
    })
    .filter(Boolean)
}

export default function LinkedInPreview({ postContent = "", postType = "post", selectedMedia = [] }) {
  // Mock user data - in real app this would come from selected account
  const mockUser = {
    name: "James Clerk",
    username: "@jamesclerk",
    avatar: "/diverse-user-avatars.png",
    verified: false,
  }

  const hashtags = extractHashtags(postContent)

  return (
    <div className="bg-white dark:bg-zinc-900 rounded-lg border border-gray-200 dark:border-zinc-800 p-4 min-h-[320px] w-full flex flex-col">
      <div className="flex items-start gap-3 mb-4">
        <img
          src={mockUser.avatar || "/placeholder.svg"}
          alt={mockUser.name}
          className="w-12 h-12 rounded-full flex-shrink-0"
        />
        <div className="flex-1 min-w-0">
          <h3 className="font-medium text-gray-900 dark:text-white">{mockUser.name}</h3>
          <p className="text-sm text-gray-500 dark:text-gray-400">Professional Title</p>
          <p className="text-xs text-gray-500 dark:text-gray-400">now</p>
        </div>
        <MoreHorizontal className="w-5 h-5 text-gray-600 dark:text-gray-400 flex-shrink-0" />
      </div>

      <div className="text-gray-900 dark:text-white mb-4 whitespace-pre-wrap break-words">
        {postContent ? highlightContent(postContent) : "Share your professional thoughts..."}
      </div>

      {hashtags.length > 0 && (
        <div className="flex flex-wrap gap-2 mb-4">
          {hashtags.map((hashtag, index) => (
            <span
              key={index}
              className="bg-blue-100 dark:bg-blue-900 text-blue-800 dark:text-blue-200 px-2 py-1 rounded-full text-sm font-medium"
            >
              {hashtag}
            </span>
          ))}
        </div>
      )}

      {selectedMedia && selectedMedia.length > 0 && (
        <div className="mb-4">
          {selectedMedia.length === 1 ? (
            <div className="rounded-lg overflow-hidden">
              <img
                src={selectedMedia[0].url || selectedMedia[0].thumbnail}
                alt={selectedMedia[0].alt || "Post image"}
                className="w-full h-64 object-cover"
              />
            </div>
          ) : (
            <div className="grid grid-cols-2 gap-1 rounded-lg overflow-hidden">
              {selectedMedia.slice(0, 4).map((media, index) => (
                <div key={media.id || index} className="relative aspect-square">
                  <img
                    src={media.url || media.thumbnail}
                    alt={media.alt || `Post image ${index + 1}`}
                    className="w-full h-full object-cover"
                  />
                  {index === 3 && selectedMedia.length > 4 && (
                    <div className="absolute inset-0 bg-black bg-opacity-60 flex items-center justify-center">
                      <span className="text-white text-lg font-semibold">+{selectedMedia.length - 4}</span>
                    </div>
                  )}
                </div>
              ))}
            </div>
          )}
        </div>
      )}

      {selectedMedia.length === 0 && (
        <div className="bg-gray-50 dark:bg-zinc-800 rounded-lg p-6 text-center text-gray-500 dark:text-gray-400 mb-4">
          No media selected
        </div>
      )}

      <div className="pt-4 border-t border-gray-100 dark:border-gray-700 mt-auto">
        <div className="flex items-center justify-around gap-1 w-full max-w-full">
          <button className="flex items-center gap-1 text-gray-600 dark:text-gray-400 hover:text-blue-600 transition-colors px-2 py-2 rounded-lg hover:bg-gray-50 dark:hover:bg-gray-700 flex-1 justify-center min-w-0">
            <ThumbsUp className="w-4 h-4 flex-shrink-0" />
          </button>
          <button className="flex items-center gap-1 text-gray-600 dark:text-gray-400 hover:text-blue-600 transition-colors px-2 py-2 rounded-lg hover:bg-gray-50 dark:hover:bg-gray-700 flex-1 justify-center min-w-0">
            <MessageCircle className="w-4 h-4 flex-shrink-0" />
          </button>
          <button className="flex items-center gap-1 text-gray-600 dark:text-gray-400 hover:text-blue-600 transition-colors px-2 py-2 rounded-lg hover:bg-gray-50 dark:hover:bg-gray-700 flex-1 justify-center min-w-0">
            <Share className="w-4 h-4 flex-shrink-0" />
          </button>
        </div>
      </div>
    </div>
  )
}
