"use client"

import { motion } from "framer-motion"
import { cn } from "@/Lib/utils"

export default function PremiumSettingsCard({ 
  title, 
  description, 
  icon, 
  children, 
  className = "",
  badge = null,
  footer = null,
  variant = "default", // default, gradient, bordered
  ...props 
}) {
  const variants = {
    default: "bg-white dark:bg-zinc-900 border border-gray-200 dark:border-zinc-800",
    gradient: "bg-gradient-to-br from-blue-50 to-purple-50 dark:from-zinc-900 dark:to-zinc-800 border border-blue-200 dark:border-zinc-700",
    bordered: "bg-white dark:bg-zinc-900 border-2 border-gray-300 dark:border-zinc-700",
    glass: "bg-white/80 dark:bg-zinc-900/80 backdrop-blur-xl border border-gray-200/50 dark:border-zinc-800/50"
  }

  return (
    <motion.div
      initial={{ opacity: 0, y: 20 }}
      animate={{ opacity: 1, y: 0 }}
      transition={{ duration: 0.3 }}
      className={cn(
        "rounded-xl shadow-sm transition-all duration-200 hover:shadow-md",
        variants[variant],
        className
      )}
      {...props}
    >
      {/* Card Header */}
      {(title || description || icon) && (
        <div className="px-4 sm:px-6 py-4 sm:py-5 border-b border-gray-100 dark:border-zinc-800">
          <div className="flex flex-col sm:flex-row sm:items-start sm:justify-between">
            <div className="flex flex-col sm:flex-row sm:items-start sm:space-x-4 space-y-3 sm:space-y-0">
              {/* Icon */}
              {icon && (
                <div className="w-10 h-10 bg-gradient-to-br from-primary to-secondary rounded-lg flex items-center justify-center text-white shadow-md">
                  {icon}
                </div>
              )}
              
              {/* Title & Description */}
              <div className="flex-1">
                <div className="flex items-center space-x-2 flex-wrap">
                  {title && (
                    <h3 className="text-lg font-semibold text-gray-900 dark:text-white">
                      {title}
                    </h3>
                  )}
                  {badge && (
                    <span className={cn(
                      "px-2 py-0.5 text-xs font-semibold rounded-full",
                      badge.type === "pro" 
                        ? "bg-gradient-to-r from-yellow-400 to-orange-500 text-white"
                        : badge.type === "new"
                        ? "bg-green-100 text-green-700 dark:bg-green-900/30 dark:text-green-400"
                        : badge.type === "beta"
                        ? "bg-purple-100 text-purple-700 dark:bg-purple-900/30 dark:text-purple-400"
                        : "bg-gray-100 text-gray-600 dark:bg-zinc-700 dark:text-gray-300"
                    )}>
                      {badge.text}
                    </span>
                  )}
                </div>
                {description && (
                  <p className="mt-1 text-sm text-gray-500 dark:text-gray-400">
                    {description}
                  </p>
                )}
              </div>
            </div>
          </div>
        </div>
      )}

      {/* Card Content */}
      <div className="px-4 sm:px-6 py-4 sm:py-6">
        {children}
      </div>

      {/* Card Footer */}
      {footer && (
        <div className="px-4 sm:px-6 py-3 sm:py-4 bg-gray-50 dark:bg-zinc-800/50 border-t border-gray-100 dark:border-zinc-800">
          {footer}
        </div>
      )}
    </motion.div>
  )
}

// Settings Row Component
export function SettingsRow({ 
  label, 
  description, 
  children, 
  required = false,
  className = "",
  labelWidth = "sm:w-1/3"
}) {
  return (
    <div className={cn("flex flex-col sm:flex-row items-start py-4", className)}>
      <div className={cn("w-full", labelWidth, "pr-0 sm:pr-8 mb-2 sm:mb-0")}>
        <label className="block text-sm font-medium text-gray-700 dark:text-gray-300">
          {label}
          {required && <span className="text-red-500 ml-1">*</span>}
        </label>
        {description && (
          <p className="mt-1 text-xs text-gray-500 dark:text-gray-400">
            {description}
          </p>
        )}
      </div>
      <div className="flex-1 w-full">
        {children}
      </div>
    </div>
  )
}

// Settings Section Component
export function SettingsSection({ 
  title, 
  children, 
  className = "",
  icon = null
}) {
  return (
    <div className={cn("mb-8", className)}>
      {title && (
        <div className="flex items-center space-x-2 mb-4">
          {icon && (
            <span className="text-gray-400 dark:text-gray-500">
              {icon}
            </span>
          )}
          <h4 className="text-sm font-semibold text-gray-400 dark:text-white uppercase tracking-wider">
            {title}
          </h4>
        </div>
      )}
      <div className="space-y-1">
        {children}
      </div>
    </div>
  )
}

// Toggle Card Component
export function ToggleCard({ 
  title, 
  description, 
  enabled, 
  onToggle,
  icon = null,
  className = ""
}) {
  return (
    <motion.div
      whileHover={{ scale: 1.01 }}
      className={cn(
        "flex items-center justify-between p-3 sm:p-4 rounded-lg border transition-all duration-200",
        enabled 
          ? "bg-blue-50 dark:bg-blue-900/20 border-blue-200 dark:border-blue-800" 
          : "bg-gray-50 dark:bg-zinc-800/50 border-gray-200 dark:border-zinc-800",
        className
      )}
    >
      <div className="flex items-start space-x-3">
        {icon && (
          <div className={cn(
            "w-8 h-8 rounded-lg flex items-center justify-center",
            enabled 
              ? "bg-blue-500 text-white" 
              : "bg-gray-300 dark:bg-zinc-700 text-gray-500 dark:text-gray-400"
          )}>
            {icon}
          </div>
        )}
        <div>
          <h5 className="text-sm font-medium text-gray-900 dark:text-white">
            {title}
          </h5>
          {description && (
            <p className="text-xs text-gray-500 dark:text-gray-400 mt-0.5">
              {description}
            </p>
          )}
        </div>
      </div>
      
      <button
        onClick={onToggle}
        className={cn(
          "relative inline-flex h-6 w-11 items-center rounded-full transition-colors",
          enabled ? "bg-blue-600" : "bg-gray-300 dark:bg-zinc-600"
        )}
      >
        <span
          className={cn(
            "inline-block h-4 w-4 transform rounded-full bg-white transition-transform",
            enabled ? "translate-x-6" : "translate-x-1"
          )}
        />
      </button>
    </motion.div>
  )
}
