"use client"

import { useState } from "react"
import { motion } from "framer-motion"

const PLACEHOLDER_TYPES = {
  post: {
    gradient: "linear-gradient(135deg, #667eea 0%, #764ba2 100%)",
    icon: (
      <svg className="w-12 h-12 text-white/80" fill="none" stroke="currentColor" viewBox="0 0 24 24">
        <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={1.5} d="M4 16l4.586-4.586a2 2 0 012.828 0L16 16m-2-2l1.586-1.586a2 2 0 012.828 0L20 14m-6-6h.01M6 20h12a2 2 0 002-2V6a2 2 0 00-2-2H6a2 2 0 00-2 2v12a2 2 0 002 2z" />
      </svg>
    ),
    text: "Social Media Post"
  },
  profile: {
    gradient: "linear-gradient(135deg, #11998e 0%, #38ef7d 100%)",
    icon: (
      <svg className="w-12 h-12 text-white/80" fill="none" stroke="currentColor" viewBox="0 0 24 24">
        <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={1.5} d="M16 7a4 4 0 11-8 0 4 4 0 018 0zM12 14a7 7 0 00-7 7h14a7 7 0 00-7-7z" />
      </svg>
    ),
    text: "Profile Image"
  },
  campaign: {
    gradient: "linear-gradient(135deg, #9f7aea 0%, #667eea 100%)",
    icon: (
      <svg className="w-12 h-12 text-white/80" fill="none" stroke="currentColor" viewBox="0 0 24 24">
        <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={1.5} d="M7 4V2a1 1 0 011-1h8a1 1 0 011 1v2m-9 0h10m-10 0l1 16h8l1-16" />
      </svg>
    ),
    text: "Campaign Visual"
  },
  gallery: {
    gradient: "linear-gradient(135deg, #fdbb2d 0%, #22c1c3 100%)",
    icon: (
      <svg className="w-12 h-12 text-white/80" fill="none" stroke="currentColor" viewBox="0 0 24 24">
        <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={1.5} d="M19 11H5m14 0a2 2 0 012 2v6a2 2 0 01-2 2H5a2 2 0 01-2-2v-6a2 2 0 012-2m14 0V9a2 2 0 00-2-2M5 11V9a2 2 0 012-2m0 0V5a2 2 0 012-2h6a2 2 0 012 2v2M7 7h10" />
      </svg>
    ),
    text: "Gallery Item"
  },
  template: {
    gradient: "linear-gradient(135deg, #74b9ff 0%, #0984e3 100%)",
    icon: (
      <svg className="w-12 h-12 text-white/80" fill="none" stroke="currentColor" viewBox="0 0 24 24">
        <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={1.5} d="M9 12h6m-6 4h6m2 5H7a2 2 0 01-2-2V5a2 2 0 012-2h5.586a1 1 0 01.707.293l5.414 5.414a1 1 0 01.293.707V19a2 2 0 01-2 2z" />
      </svg>
    ),
    text: "Template Design"
  },
  analytics: {
    gradient: "linear-gradient(135deg, #ffecd2 0%, #fcb69f 100%)",
    icon: (
      <svg className="w-12 h-12 text-white/80" fill="none" stroke="currentColor" viewBox="0 0 24 24">
        <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={1.5} d="M9 19v-6a2 2 0 00-2-2H5a2 2 0 00-2 2v6a2 2 0 002 2h2a2 2 0 002-2zm0 0V9a2 2 0 012-2h2a2 2 0 012 2v10m-6 0a2 2 0 002 2h2a2 2 0 002-2m0 0V5a2 2 0 012-2h2a2 2 0 012 2v14a2 2 0 01-2 2h-2a2 2 0 01-2-2z" />
      </svg>
    ),
    text: "Analytics Chart"
  }
}

export default function ProfessionalImage({ 
  src, 
  alt, 
  type = "post", 
  className = "", 
  width = "100%", 
  height = "200px",
  showText = true,
  animated = true,
  ...props 
}) {
  const [imageError, setImageError] = useState(false)
  const [imageLoaded, setImageLoaded] = useState(false)

  const placeholder = PLACEHOLDER_TYPES[type] || PLACEHOLDER_TYPES.post

  const handleImageError = () => {
    setImageError(true)
  }

  const handleImageLoad = () => {
    setImageLoaded(true)
  }

  if (!src || imageError) {
    return (
      <motion.div
        initial={animated ? { opacity: 0, scale: 0.9 } : false}
        animate={animated ? { opacity: 1, scale: 1 } : false}
        transition={{ duration: 0.3 }}
        className={`relative flex flex-col items-center justify-center rounded-lg overflow-hidden ${className}`}
        style={{ 
          width, 
          height, 
          background: placeholder.gradient 
        }}
        {...props}
      >
        {/* Background Pattern */}
        <div className="absolute inset-0 opacity-10">
          <div className="w-full h-full" style={{
            backgroundImage: `url("data:image/svg+xml,%3Csvg width='40' height='40' viewBox='0 0 40 40' xmlns='http://www.w3.org/2000/svg'%3E%3Cg fill='%23ffffff' fill-opacity='0.1'%3E%3Cpath d='M20 20c0-11.046-8.954-20-20-20s-20 8.954-20 20 8.954 20 20 20 20-8.954 20-20zm-30 0c0-5.523 4.477-10 10-10s10 4.477 10 10-4.477 10-10 10-10-4.477-10-10z'/%3E%3C/g%3E%3C/svg%3E")`,
            backgroundSize: '40px 40px'
          }} />
        </div>
        
        {/* Content */}
        <div className="relative z-10 flex flex-col items-center justify-center text-center p-6">
          {/* Icon */}
          <motion.div
            animate={animated ? { 
              scale: [1, 1.1, 1],
              rotate: [0, 5, -5, 0] 
            } : false}
            transition={{ 
              duration: 2,
              repeat: Infinity,
              repeatType: "reverse"
            }}
            className="mb-3"
          >
            {placeholder.icon}
          </motion.div>
          
          {/* Text */}
          {showText && (
            <div className="text-white/90">
              <p className="text-sm font-medium mb-1">{placeholder.text}</p>
              <p className="text-xs text-white/70">No image available</p>
            </div>
          )}
        </div>

        {/* Decorative elements */}
        <div className="absolute top-2 right-2 w-8 h-8 border border-white/20 rounded-full flex items-center justify-center">
          <div className="w-2 h-2 bg-white/40 rounded-full" />
        </div>
        
        <div className="absolute bottom-2 left-2 w-6 h-6 border border-white/20 rounded-lg flex items-center justify-center">
          <div className="w-1.5 h-1.5 bg-white/40 rounded-sm" />
        </div>
      </motion.div>
    )
  }

  return (
    <div className={`relative overflow-hidden ${className}`} style={{ width, height }} {...props}>
      {/* Loading placeholder */}
      {!imageLoaded && (
        <div 
          className="absolute inset-0 flex items-center justify-center"
          style={{ background: placeholder.gradient }}
        >
          <motion.div
            animate={{ rotate: 360 }}
            transition={{ duration: 1, repeat: Infinity, ease: "linear" }}
            className="w-8 h-8 border-2 border-white/30 border-t-white rounded-full"
          />
        </div>
      )}
      
      {/* Actual image */}
      <motion.img
        src={src}
        alt={alt}
        onError={handleImageError}
        onLoad={handleImageLoad}
        initial={animated ? { opacity: 0 } : false}
        animate={animated && imageLoaded ? { opacity: 1 } : false}
        transition={{ duration: 0.5 }}
        className={`w-full h-full object-cover ${imageLoaded ? 'opacity-100' : 'opacity-0'}`}
        {...props}
      />
    </div>
  )
}

// Utility component for common image placeholders
export const ImagePlaceholder = ({ type, className = "", ...props }) => (
  <ProfessionalImage
    src=""
    type={type}
    className={className}
    {...props}
  />
)