"use client"

import { useState } from "react"
import { router } from "@inertiajs/react"
import { Button } from "../../Components/UI/Button"
import { Card, CardContent } from "../../Components/UI/Card"
import {
  CheckCircle2,
  ArrowRight,
  Rocket,
  Copy,
  ExternalLink,
  Book,
  Users,
  Settings,
  Shield
} from "lucide-react"
import { motion } from "framer-motion"
import Confetti from 'react-confetti'

export default function Complete({ 
  appName = "Postuno",
  adminUrl = "/admin",
  loginUrl = "/login",
  credentials = {}
}) {
  const [copiedEmail, setCopiedEmail] = useState(false)
  const [copiedPassword, setCopiedPassword] = useState(false)
  const [showConfetti, setShowConfetti] = useState(true)

  const handleCopy = async (text, type) => {
    try {
      // Try modern clipboard API first
      if (navigator.clipboard && window.isSecureContext) {
        await navigator.clipboard.writeText(text)
      } else {
        // Fallback for older browsers or non-secure contexts
        const textArea = document.createElement('textarea')
        textArea.value = text
        textArea.style.position = 'fixed'
        textArea.style.left = '-999999px'
        textArea.style.top = '-999999px'
        document.body.appendChild(textArea)
        textArea.focus()
        textArea.select()
        document.execCommand('copy')
        textArea.remove()
      }
      
      if (type === 'email') {
        setCopiedEmail(true)
        setTimeout(() => setCopiedEmail(false), 2000)
      } else {
        setCopiedPassword(true)
        setTimeout(() => setCopiedPassword(false), 2000)
      }
    } catch (err) {
      console.error('Failed to copy text: ', err)
      // Show error message or fallback behavior
      alert('Copy failed. Please manually select and copy the text.')
    }
  }

  const handleGoToDashboard = () => {
    router.visit(loginUrl)
  }

  // Hide confetti after 5 seconds
  setTimeout(() => setShowConfetti(false), 5000)

  const quickLinks = [
    { icon: Settings, title: "Configure Settings", desc: "Customize your application" },
    { icon: Users, title: "Manage Users", desc: "Add team members and users" },
    { icon: Shield, title: "Security Settings", desc: "Configure security options" },
    { icon: Book, title: "Documentation", desc: "Learn more about features" }
  ]

  return (
    <div className="min-h-screen bg-gradient-to-br from-blue-50 via-white to-purple-50 dark:from-gray-900 dark:via-gray-800 dark:to-gray-900">
      {showConfetti && typeof window !== 'undefined' && (
        <Confetti
          width={window.innerWidth}
          height={window.innerHeight}
          recycle={false}
          numberOfPieces={200}
        />
      )}
      
      <div className="container mx-auto px-4 py-12">
        <div className="max-w-4xl mx-auto">
          {/* Header */}
          <motion.div 
            initial={{ opacity: 0, scale: 0.9 }}
            animate={{ opacity: 1, scale: 1 }}
            transition={{ duration: 0.5 }}
            className="text-center mb-12"
          >
            <motion.div 
              initial={{ scale: 0 }}
              animate={{ scale: 1 }}
              transition={{ delay: 0.2, type: "spring", stiffness: 200 }}
              className="inline-flex items-center justify-center w-24 h-24 rounded-full bg-gradient-to-r from-green-400 to-blue-500 mb-6"
            >
              <CheckCircle2 className="w-12 h-12 text-white" />
            </motion.div>
            <h1 className="text-4xl font-bold text-gray-900 dark:text-white mb-4">
              Installation Complete!
            </h1>
            <p className="text-lg text-gray-600 dark:text-gray-400">
              {appName} has been successfully installed and configured
            </p>
          </motion.div>

          {/* Progress Bar - Complete */}
          <div className="mb-8">
            <div className="flex items-center justify-between mb-2">
              <span className="text-sm text-gray-600 dark:text-gray-400">Installation Progress</span>
              <span className="text-sm font-semibold text-green-600 dark:text-green-400">100% Complete!</span>
            </div>
            <div className="w-full bg-gray-200 rounded-full h-2 dark:bg-gray-700">
              <motion.div
                initial={{ width: '75%' }}
                animate={{ width: '100%' }}
                transition={{ duration: 1, delay: 0.5 }}
                className="bg-gradient-to-r from-green-400 to-blue-500 h-2 rounded-full"
              />
            </div>
          </div>

          {/* Main Card */}
          <motion.div
            initial={{ opacity: 0, y: 20 }}
            animate={{ opacity: 1, y: 0 }}
            transition={{ delay: 0.3 }}
          >
            <Card className="backdrop-blur-sm bg-white/90 dark:bg-gray-800/90 shadow-xl border-0">
              <CardContent className="p-8">
                {/* Success Message */}
                <div className="bg-green-50 dark:bg-green-900/20 border border-green-200 dark:border-green-800 rounded-lg p-6 mb-8">
                  <div className="flex items-start">
                    <Rocket className="w-6 h-6 text-green-500 mt-0.5 mr-3" />
                    <div className="flex-1">
                      <h3 className="text-green-800 dark:text-green-300 font-semibold text-lg mb-2">
                        Your Application is Ready!
                      </h3>
                      <p className="text-green-600 dark:text-green-400 text-sm">
                        All systems are operational. You can now start using {appName} to manage your social media content.
                      </p>
                    </div>
                  </div>
                </div>

                {/* Login Credentials */}
                {credentials.email && (
                  <div className="mb-8">
                    <h3 className="font-semibold text-gray-900 dark:text-white mb-4">
                      Your Login Credentials
                    </h3>
                    <div className="bg-gray-50 dark:bg-gray-900/50 rounded-lg p-4 space-y-3">
                      <div className="flex items-center justify-between">
                        <div>
                          <p className="text-sm text-gray-500 dark:text-gray-400">Email</p>
                          <p className="font-mono text-gray-900 dark:text-white">{credentials.email}</p>
                        </div>
                        <Button
                          size="sm"
                          variant="outline"
                          onClick={() => handleCopy(credentials.email, 'email')}
                        >
                          {copiedEmail ? (
                            <CheckCircle2 className="w-4 h-4" />
                          ) : (
                            <Copy className="w-4 h-4" />
                          )}
                        </Button>
                      </div>
                      {credentials.password && (
                        <div className="flex items-center justify-between">
                          <div>
                            <p className="text-sm text-gray-500 dark:text-gray-400">Password</p>
                            <p className="font-mono text-gray-900 dark:text-white">••••••••</p>
                          </div>
                          <Button
                            size="sm"
                            variant="outline"
                            onClick={() => handleCopy(credentials.password, 'password')}
                          >
                            {copiedPassword ? (
                              <CheckCircle2 className="w-4 h-4" />
                            ) : (
                              <Copy className="w-4 h-4" />
                            )}
                          </Button>
                        </div>
                      )}
                    </div>
                    <p className="text-xs text-gray-500 dark:text-gray-400 mt-2">
                      Please save these credentials securely. The password won't be shown again.
                    </p>
                  </div>
                )}

                {/* Important Notes */}
                <div className="bg-blue-50 dark:bg-blue-900/20 border border-blue-200 dark:border-blue-800 rounded-lg p-4 mb-8">
                  <h4 className="font-medium text-blue-800 dark:text-blue-300 mb-2">
                    Important Security Notes
                  </h4>
                  <ul className="text-sm text-blue-600 dark:text-blue-400 space-y-1">
                    <li>• Do not share envato purchase key for security</li>
                    <li>• Set proper file permissions (755 for directories, 644 for files)</li>
                    <li>• Configure SSL certificate for secure connections</li>
                    <li>• Regular backups are recommended</li>
                  </ul>
                </div>

                {/* Action Button */}
                <Button
                  onClick={handleGoToDashboard}
                  className="w-full h-12 text-lg font-semibold bg-gradient-to-r from-green-500 to-blue-600 hover:from-green-600 hover:to-blue-700 text-white"
                >
                  <span className="flex items-center justify-center">
                    Go to Dashboard
                    <ArrowRight className="ml-2 h-5 w-5" />
                  </span>
                </Button>

                {/* Footer Links */}
                <div className="flex justify-center gap-6 mt-6">
                  <a href="https://support.igensolutionsltd.com" target="_blank" className="text-sm text-gray-600 dark:text-gray-400 hover:text-gray-900 dark:hover:text-white flex items-center">
                    <Book className="w-4 h-4 mr-1" />
                    Documentation
                  </a>
                  <a href="https://support.igensolutionsltd.com/login" target="_blank" className="text-sm text-gray-600 dark:text-gray-400 hover:text-gray-900 dark:hover:text-white flex items-center">
                    <Users className="w-4 h-4 mr-1" />
                    Support
                  </a> 
                </div>
              </CardContent>
            </Card>
          </motion.div>
        </div>
      </div>
    </div>
  )
}