"use client"

import { useState , useEffect } from "react"
import { Input } from "@/Components/UI/Input"
import { Button } from "@/Components/UI/Button"
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/Components/UI/Select"
import { router } from "@inertiajs/react"
import useTranslate from "@/hooks/useTranslate";

const ImageModule = ({ imageAiSetting, settingConfig }) => {

  const t = useTranslate()
  const [isSubmitting, setIsSubmitting] = useState(false)


  const [imageData, setImageData] = useState({
    module: "image_ai",
    provider: "open_ai",
    config: {
      model: "",
      default_resolution: "",
      api_key: ""
    }
  })

  useEffect(() => {
    if (imageAiSetting) {
      setImageData(imageAiSetting)
    }
  }, [imageAiSetting])

  const handleInputChange = (field, value) => {
    setImageData((prev) => ({
      ...prev,
      config: {
        ...prev.config,
        [field]: value,
      }
    }))
  }

  const handleSubmit = () => {
    setIsSubmitting(true)

    router.post(route('user.ai_setting.store'), imageData, {
      onFinish: () => setIsSubmitting(false),
      onError: () => setIsSubmitting(false),
    })
  }

  return (
    <div className="space-y-8">
      {/* Header */}
      <div className="flex items-center justify-between mb-6">
        <div>
          <h3 className="text-lg font-semibold text-gray-900 dark:text-gray-100">{t("Image AI Configuration")}</h3>
          <p className="text-sm text-gray-500 dark:text-gray-400 mt-1">
            {t("Configure DALL-E image generation settings and API access")}
          </p>
        </div>
        <Button
          isLoading={isSubmitting}
          onClick={handleSubmit}
          className="bg-gradient-to-r from-primary to-secondary hover:from-primary/90 hover:to-secondary/90 text-white px-6 py-2.5 rounded-lg font-medium shadow-sm transition-all duration-200 flex items-center gap-2"
        >
          {!isSubmitting && (
            <svg className="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
              <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M5 13l4 4L19 7" />
            </svg>
          )}
          {isSubmitting ? t("Saving...") : t("Save Configuration")}
        </Button>
      </div>

      {/* Model Information Cards */}
      <div className="grid lg:grid-cols-2 grid-cols-1 lg:gap-6 gap-4">
        {/* DALL-E 2 Card */}
        <div className="bg-gray-50 dark:bg-zinc-800 dark:border-none dark:text-neutral-200 border border-gray-200 rounded-lg p-6">
          <h3 className="text-lg font-semibold mb-3">{t("DALL-E 2")}</h3>
          <div>
            <h4 className="text-sm font-medium text-gray-700 dark:text-gray-400 mb-2">{t("Supported Resolutions:")}</h4>
            <p className="text-sm text-gray-600 dark:text-gray-400">256x256 (square), 512x512 (square), 1024x1024 (square)</p>
          </div>
        </div>

        {/* DALL-E 3 Card */}
        <div className="bg-gray-50 dark:bg-zinc-800 dark:border-none dark:text-neutral-200 border border-gray-200 rounded-lg p-6">
          <h3 className="text-lg font-semibold mb-3">{t("DALL-E 3")}</h3>
          <div>
            <h4 className="text-sm font-medium text-gray-700 mb-2 dark:text-gray-400">{t("Supported Resolutions:")}</h4>
            <p className="text-sm text-gray-600 dark:text-gray-400">1024x1024 (square), 1792x1024 (horizontal), 1024x1792 (vertical)</p>
          </div>
        </div>
      </div>

      {/* Configuration Form */}
      <form onSubmit={handleSubmit} className="space-y-6">
        <div className="grid lg:grid-cols-2 grid-cols-1 lg:gap-6 gap-4">
          {/* Image Generation Model */}
          <div>
            <label className="block text-sm font-medium text-gray-700 mb-2">{t("Image Generation Model")}</label>
            <Select value={imageData.config.model} onValueChange={(value) => handleInputChange("model", value)}>
              <SelectTrigger className="w-full bg-gray-50 border-gray-200">
                <SelectValue placeholder="Select model" />
              </SelectTrigger>
              <SelectContent>
                {Object.entries(settingConfig.image_ai.providers.openai.models).map(([value, label]) => (
                  <SelectItem key={value} value={value}>
                    {label}
                  </SelectItem>
                ))}
              </SelectContent>
            </Select>
          </div>

          {/* Default Resolution */}
          <div>
            <label className="block text-sm font-medium text-gray-700 mb-2">{t("Default Resolution")}</label>
            <Select
              value={imageData.config.default_resolution}
              onValueChange={(value) => handleInputChange("default_resolution", value)}
            >
              <SelectTrigger className="w-full bg-gray-50 border-gray-200">
                <SelectValue placeholder="Select resolution" />
              </SelectTrigger>
              <SelectContent>
                {Object.entries(settingConfig.image_ai.settings.default_resolution).map(([value, label]) => (
                  <SelectItem key={value} value={value}>
                    {label}
                  </SelectItem>
                ))}
              </SelectContent>
            </Select>
          </div>
        </div>

        {/* Open API Secret Key */}
        <div>
          <label className="block text-sm font-medium text-gray-700 mb-2">{t("Open API Secret Key")}</label>
          <Input
            type="text"
            placeholder="Enter your OpenAI API key (sk-...)"
            value={imageData.config.api_key}
            onChange={(e) => handleInputChange("api_key", e.target.value)}
            className="bg-gray-50 border-gray-200"
          />
        </div>
      </form>
    </div>
  )
}

export default ImageModule
