"use client"

import { useState, useEffect } from "react"
import { Button } from "@/Components/UI/Button"
import { Input } from "@/Components/UI/Input"
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/Components/UI/Select"
import { router } from "@inertiajs/react"
import useTranslate from "@/hooks/useTranslate";

export default function VideoModule({ videoAiSetting, settingConfig }) {
  const t = useTranslate()

  const [isSubmitting, setIsSubmitting] = useState(false)

  const [videoData, setVideoData] = useState({
    module: "video_ai",
    provider: "kling_ai",
    config: {
      model: "",
      default_resolution: "",
      api_key: ""
    }
  })

  useEffect(() => {
    if (videoAiSetting) {
      setVideoData(videoAiSetting)
    }
  }, [videoAiSetting])

  const handleInputChange = (field, value) => {
    setVideoData((prev) => ({
      ...prev,
      config: {
        ...prev.config,
        [field]: value,
      }
    }))
  }

  const handleSubmit = () => {
    setIsSubmitting(true)

    router.post(route('user.ai_setting.store'), videoData, {
      onFinish: () => setIsSubmitting(false),
      onError: () => setIsSubmitting(false),
    })
  }

  return (
    <div className="space-y-8">
      {/* Header */}
      <div className="flex items-center justify-between mb-6">
        <div>
          <h3 className="text-lg font-semibold text-gray-900 dark:text-gray-100">{t("Video AI Configuration")}</h3>
          <p className="text-sm text-gray-500 dark:text-gray-400 mt-1">
            {t("Configure Kling AI video generation settings and API access")}
          </p>
        </div>
        <Button
          isLoading={isSubmitting}
          onClick={handleSubmit}
          className="bg-gradient-to-r from-primary to-secondary hover:from-primary/90 hover:to-secondary/90 text-white px-6 py-2.5 rounded-lg font-medium shadow-sm transition-all duration-200 flex items-center gap-2"
        >
          {!isSubmitting && (
            <svg className="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
              <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M5 13l4 4L19 7" />
            </svg>
          )}
          {isSubmitting ? t("Saving...") : t("Save Configuration")}
        </Button>
      </div>

      {/* Note Section */}
      <div className="bg-blue-50 dark:bg-zinc-800 dark:border-none dark:text-neutral-200 border border-blue-200 rounded-lg p-6">
        <div className="space-y-3">
          <h3 className="text-primary dark:text-neutral-200 font-semibold text-sm">{t("Note:")}</h3>
          <p className="text-gray-600 dark:text-gray-400 text-sm leading-relaxed">
            {t("Kling AI Generates High-quality Videos From Text Prompts. For Best Results, Provide Clear, Descriptive Prompts. Default Output Is 1080p With Durations Up To 5-10 Seconds.")}
          </p>
          <p className="text-gray-700 dark:text-gray-200 text-sm">
            <span className="font-semibold">{t("Model Used:")}</span> {t("Kling V1")}
          </p>
        </div>
      </div>

      {/* Form Section */}
      <form onSubmit={handleSubmit} className="space-y-6">
        {/* Two Column Layout */}
        <div className="grid lg:grid-cols-2 grid-cols-1 lg:gap-6 gap-4">
          {/* Video Generation Model */}
          <div className="space-y-2">
            <label className="block text-sm font-medium text-gray-700">{t("Video Generation Model")}</label>
            <Select
              value={videoData.config.model}
              onValueChange={(value) => handleInputChange("model", value)}
            >
              <SelectTrigger className="w-full bg-gray-50 border-gray-200">
                <SelectValue placeholder="Select model" />
              </SelectTrigger>
              <SelectContent>
                {Object.entries(settingConfig.video_ai.providers.kling_ai.models).map(([value, label]) => (
                  <SelectItem key={value} value={value}>
                    {label}
                  </SelectItem>
                ))}
              </SelectContent>
            </Select>
          </div>

          {/* Default Aspect Ratio */}
          <div className="space-y-2">
            <label className="block text-sm font-medium text-gray-700">{t("Default Aspect Ratio")}</label>
            <Select
              value={videoData.config.default_aspect_ratio}
              onValueChange={(value) => handleInputChange("default_aspect_ratio", value)}
            >
              <SelectTrigger className="w-full bg-gray-50 border-gray-200">
                <SelectValue placeholder="Select aspect ratio" />
              </SelectTrigger>
              <SelectContent>
                {Object.entries(settingConfig.video_ai.settings.default_aspect_ratio).map(([value, label]) => (
                  <SelectItem key={value} value={value}>
                    {label}
                  </SelectItem>
                ))}
              </SelectContent>
            </Select>
          </div>
        </div>

        {/* API Key - Full Width */}
        <div className="space-y-2">
          <label className="block text-sm font-medium text-gray-700">{t("Kling API Secret Key")}</label>
          <Input
            type="text"
            placeholder={t("Kling API Secret Key")}
            value={videoData.config.api_key}
            onChange={(e) => handleInputChange("api_key", e.target.value)}
            className="bg-gray-50 border-gray-200"
          />
        </div>
      </form>
    </div>
  )
}
