"use client"

import { ChevronLeft, ChevronRight } from "lucide-react"
import { Swiper, SwiperSlide } from "swiper/react"
import { Navigation } from "swiper/modules"
import "swiper/css"
import "swiper/css/navigation"

import PlatformIcon from "@/Components/UI/PlatformIcon"
import PostReaction from "@/Components/UI/PostReaction"
import { Avatar, AvatarFallback, AvatarImage } from "@/Components/UI/Avatar"

export default function PostCard({ data = [] }) {
  const posts = data.length > 0 ? data : []

  return (
    <div className="bg-white dark:bg-zinc-900 rounded-lg shadow-sm border border-gray-200 dark:border-zinc-800">
      {/* Header with title and navigation */}
      <div className="flex items-center justify-between p-6 pb-0">
        <div>
          <h3 className="text-lg font-semibold text-gray-900 dark:text-white">Top performing post</h3>
          <p className="text-sm text-gray-600 dark:text-gray-400 mt-1">
            View your best performing posts from the reporting
          </p>
        </div>
        {posts.length > 0 && (
          <div className="flex space-x-2">
            <button className="swiper-button-prev-custom w-8 h-8 rounded-full border flex items-center justify-center text-gray-600 dark:text-gray-400 hover:bg-gray-50 dark:hover:bg-zinc-800">
              <ChevronLeft className="w-4 h-4" />
            </button>
            <button className="swiper-button-next-custom w-8 h-8 rounded-full border flex items-center justify-center text-gray-600 dark:text-gray-400 hover:bg-gray-50 dark:hover:bg-zinc-800">
              <ChevronRight className="w-4 h-4" />
            </button>
          </div>
        )}
      </div>

      {posts.length === 0 ? (
        <div className="p-6 pt-4 h-[26rem] flex items-center justify-center">
          <p className="text-gray-500 dark:text-gray-400 text-sm">
            There is no post available, Top performing post will appear here
          </p>
        </div>
      ) : (
        <div className="p-6 pt-4">
          <Swiper
            modules={[Navigation]}
            navigation={{
              nextEl: ".swiper-button-next-custom",
              prevEl: ".swiper-button-prev-custom",
            }}
            spaceBetween={16}
            breakpoints={{
              0: { slidesPerView: 1 },
              640: { slidesPerView: 2 }, // mobile
              768: { slidesPerView: 2 },   // tablet
              1024: { slidesPerView: 3 },  // desktop
              1280: { slidesPerView: 3 },  // large desktop
            }}
          >
            {posts.map((post) => (
              <SwiperSlide key={post.id}>
                <div className="space-y-3 p-3 rounded-lg border border-gray-100 dark:border-zinc-800 hover:border-gray-200 dark:hover:border-zinc-700 hover:shadow-sm transition-all duration-200 h-full">
                  {/* Author info */}
                  <div className="flex items-center space-x-2">
                    <Avatar className="h-10 w-10">
                      <AvatarImage src={post?.account?.avatar} alt="User Avatar" />
                      <AvatarFallback>AV</AvatarFallback>
                    </Avatar>
                    <div className="flex items-center space-x-1">
                      <span className="text-sm font-medium text-gray-900 dark:text-white">{post.author}</span>
                      <PlatformIcon slug={post.platform?.slug} size={14} round={true} />
                    </div>
                  </div>

                  {/* Publish date */}
                  <p className="text-xs text-gray-500 dark:text-gray-400">Published {post.publishDate}</p>

                  {/* Post image */}
                  <div className="aspect-video rounded-lg overflow-hidden bg-gray-100 dark:bg-zinc-800">
                    {post.media && post.media.length > 0 && (
                      post.media[0].isVideo ? (
                        <video
                          src={post.media[0].thumbnail}
                          muted
                          playsInline
                          loop
                          autoPlay
                          className="object-cover w-full h-full"
                        />
                      ) : (
                        <img
                          src={post.media[0].thumbnail || "/placeholder.svg"}
                          alt={post.media[0].alt || post.title}
                          className="object-cover w-full h-full"
                        />
                      )
                    )}
                  </div>

                  {/* Post content */}
                  <p className="text-sm text-gray-700 dark:text-gray-300 line-clamp-2">{post.content}</p>

                  {/* Engagement */}
                  <div className="space-y-2">
                    <div className="flex items-center justify-between">
                      <div className="flex items-center space-x-2">
                        <div className="flex space-x-0 -ml-1">
                          <PostReaction reactKey="like" />
                          <PostReaction reactKey="love" />
                          <PostReaction reactKey="care" />
                        </div>
                        <span className="text-sm text-gray-600 dark:text-gray-400">{post.engagement.reactions}</span>
                      </div>
                      <div className="flex items-center space-x-3">
                        <div className="flex items-center space-x-1">
                          <span className="text-sm text-gray-600 dark:text-gray-400">{post.engagement.comments}</span>
                          <PostReaction reactKey="fbcomment" />
                        </div>
                        <div className="flex items-center space-x-1">
                          <span className="text-sm text-gray-600 dark:text-gray-400">{post.engagement.shares}</span>
                          <PostReaction reactKey="fbshare" />
                        </div>
                      </div>
                    </div>
                  </div>

                  {/* Analytics */}
                  <div className="space-y-2 pt-2 border-t border-gray-200 dark:border-zinc-800">
                    <div className="flex justify-between items-center">
                      <span className="text-xs text-gray-500 dark:text-gray-400">Impressions</span>
                      <span className="text-sm font-medium text-gray-900 dark:text-white">
                        {post.analytics.impressions}
                      </span>
                    </div>
                    <div className="flex justify-between items-center">
                      <span className="text-xs text-gray-500 dark:text-gray-400">Engagement</span>
                      <span className="text-sm font-medium text-gray-900 dark:text-white">
                        {post.analytics.engagement}
                      </span>
                    </div>
                  </div>
                </div>
              </SwiperSlide>
            ))}
          </Swiper>
        </div>
      )}
    </div>
  )
}
