"use client"

import { useState } from "react"
import { motion } from "framer-motion"
import ProfessionalAvatar, { AvatarGroup } from "@/Components/UI/ProfessionalAvatar"
import ProfessionalImage, { ImagePlaceholder } from "@/Components/UI/ProfessionalImage"
import PremiumPreloader from "@/Components/PremiumPreloader"
import ThemeColorTemplates from "@/Components/AppSettings/ThemeColorTemplates"

export default function ProfessionalShowcase() {
  const [showPreloader, setShowPreloader] = useState(false)
  const [currentTheme, setCurrentTheme] = useState({
    primary_color: "#667eea",
    secondary_color: "#764ba2"
  })

  // Demo data
  const demoAvatars = [
    { name: "John Doe", email: "john@example.com" },
    { name: "Jane Smith", email: "jane@example.com" },
    { name: "Mike Johnson", email: "mike@example.com" },
    { name: "Sarah Wilson", email: "sarah@example.com" },
    { name: "Alex Brown", email: "alex@example.com" },
    { name: "Emma Davis", email: "emma@example.com" }
  ]

  return (
    <div className="space-y-12 p-8 bg-gray-50 dark:bg-gray-900 min-h-screen">
      <div className="max-w-6xl mx-auto">
        
        {/* Header */}
        <div className="text-center mb-12">
          <h1 className="text-4xl font-bold text-gray-900 dark:text-white mb-4">
            Professional Postuno Components
          </h1>
          <p className="text-lg text-gray-600 dark:text-gray-400 max-w-3xl mx-auto">
            Experience the new professional design system with modern components, 
            beautiful avatars, smart image placeholders, and premium loading experiences.
          </p>
        </div>

        {/* Professional Avatars Section */}
        <motion.section
          initial={{ opacity: 0, y: 20 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ delay: 0.1 }}
          className="bg-white dark:bg-zinc-900 rounded-xl p-8 shadow-lg"
        >
          <h2 className="text-2xl font-bold mb-6 text-gray-900 dark:text-white">
            Professional Avatar System
          </h2>
          
          <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-8">
            {/* Initials Avatars */}
            <div>
              <h3 className="text-lg font-semibold mb-4 text-gray-700 dark:text-gray-300">
                Initials Style
              </h3>
              <div className="flex items-center space-x-3">
                {demoAvatars.slice(0, 3).map((user, idx) => (
                  <ProfessionalAvatar
                    key={idx}
                    name={user.name}
                    email={user.email}
                    type="initials"
                    size="lg"
                  />
                ))}
              </div>
            </div>

            {/* Emoji Avatars */}
            <div>
              <h3 className="text-lg font-semibold mb-4 text-gray-700 dark:text-gray-300">
                Emoji Style
              </h3>
              <div className="flex items-center space-x-3">
                {['business', 'creative', 'tech'].map((category, idx) => (
                  <ProfessionalAvatar
                    key={idx}
                    name={demoAvatars[idx].name}
                    email={demoAvatars[idx].email}
                    type="emoji"
                    emojiCategory={category}
                    size="lg"
                  />
                ))}
              </div>
            </div>

            {/* Pattern Avatars */}
            <div>
              <h3 className="text-lg font-semibold mb-4 text-gray-700 dark:text-gray-300">
                Pattern Style
              </h3>
              <div className="flex items-center space-x-3">
                {demoAvatars.slice(0, 3).map((user, idx) => (
                  <ProfessionalAvatar
                    key={idx}
                    name={user.name}
                    email={user.email}
                    type="pattern"
                    size="lg"
                    showOnlineStatus
                    isOnline={idx % 2 === 0}
                  />
                ))}
              </div>
            </div>
          </div>

          {/* Avatar Groups */}
          <div className="mt-8 pt-8 border-t border-gray-200 dark:border-zinc-800">
            <h3 className="text-lg font-semibold mb-4 text-gray-700 dark:text-gray-300">
              Avatar Groups
            </h3>
            <div className="flex items-center space-x-8">
              <div>
                <p className="text-sm text-gray-500 mb-2">Team (6 members)</p>
                <AvatarGroup avatars={demoAvatars} max={4} size="md" />
              </div>
              <div>
                <p className="text-sm text-gray-500 mb-2">Small Group (3 members)</p>
                <AvatarGroup avatars={demoAvatars.slice(0, 3)} size="lg" />
              </div>
            </div>
          </div>
        </motion.section>

        {/* Professional Images Section */}
        <motion.section
          initial={{ opacity: 0, y: 20 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ delay: 0.2 }}
          className="bg-white dark:bg-zinc-900 rounded-xl p-8 shadow-lg"
        >
          <h2 className="text-2xl font-bold mb-6 text-gray-900 dark:text-white">
            Smart Image Placeholders
          </h2>
          
          <div className="grid grid-cols-2 md:grid-cols-3 lg:grid-cols-6 gap-4">
            {['post', 'profile', 'campaign', 'gallery', 'template', 'analytics'].map(type => (
              <div key={type}>
                <p className="text-sm text-gray-500 mb-2 capitalize">{type}</p>
                <ImagePlaceholder
                  type={type}
                  width="100%"
                  height="120px"
                  className="rounded-lg"
                />
              </div>
            ))}
          </div>

          <div className="mt-8 pt-8 border-t border-gray-200 dark:border-zinc-800">
            <h3 className="text-lg font-semibold mb-4 text-gray-700 dark:text-gray-300">
              With Loading States
            </h3>
            <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
              <ProfessionalImage
                src="https://via.placeholder.com/400x200"
                alt="Sample Image"
                type="campaign"
                width="100%"
                height="200px"
                className="rounded-lg"
              />
              <ProfessionalImage
                src="" // Intentionally broken to show fallback
                alt="Broken Image"
                type="gallery"
                width="100%"
                height="200px"
                className="rounded-lg"
              />
            </div>
          </div>
        </motion.section>

        {/* Premium Preloader Section */}
        <motion.section
          initial={{ opacity: 0, y: 20 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ delay: 0.3 }}
          className="bg-white dark:bg-zinc-900 rounded-xl p-8 shadow-lg"
        >
          <h2 className="text-2xl font-bold mb-6 text-gray-900 dark:text-white">
            Premium Preloader
          </h2>
          
          <div className="text-center">
            <p className="text-gray-600 dark:text-gray-400 mb-6">
              Experience our beautiful loading animation with progress tracking
            </p>
            
            <button
              onClick={() => setShowPreloader(true)}
              className="px-6 py-3 bg-gradient-to-r from-blue-500 to-purple-600 text-white rounded-lg font-medium hover:shadow-lg transition-all duration-200 transform hover:scale-105"
            >
              Show Premium Preloader
            </button>
          </div>

          {showPreloader && (
            <PremiumPreloader
              siteName="Postuno"
              onComplete={() => setShowPreloader(false)}
            />
          )}
        </motion.section>

        {/* Theme Templates Section */}
        <motion.section
          initial={{ opacity: 0, y: 20 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ delay: 0.4 }}
          className="bg-white dark:bg-zinc-900 rounded-xl p-8 shadow-lg"
        >
          <h2 className="text-2xl font-bold mb-6 text-gray-900 dark:text-white">
            Professional Theme Templates
          </h2>
          
          <ThemeColorTemplates
            currentTheme={currentTheme}
            onTemplateSelect={setCurrentTheme}
          />

          <div className="mt-8 pt-8 border-t border-gray-200 dark:border-zinc-800">
            <h3 className="text-lg font-semibold mb-4 text-gray-700 dark:text-gray-300">
              Current Theme Preview
            </h3>
            <div 
              className="h-20 rounded-lg flex items-center justify-center text-white font-semibold text-xl"
              style={{
                background: `linear-gradient(135deg, ${currentTheme.primary_color} 0%, ${currentTheme.secondary_color} 100%)`
              }}
            >
              {currentTheme.primary_color} → {currentTheme.secondary_color}
            </div>
          </div>
        </motion.section>

        {/* Features Summary */}
        <motion.section
          initial={{ opacity: 0, y: 20 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ delay: 0.5 }}
          className="bg-gradient-to-r from-blue-500 to-purple-600 rounded-xl p-8 text-white"
        >
          <h2 className="text-2xl font-bold mb-6">✨ Professional Features Implemented</h2>
          
          <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
            <div className="space-y-4">
              <div className="flex items-start space-x-3">
                <div className="w-6 h-6 bg-white/20 rounded-full flex items-center justify-center mt-0.5">
                  <span className="text-sm">✓</span>
                </div>
                <div>
                  <h3 className="font-semibold">Professional Theme Templates</h3>
                  <p className="text-blue-100 text-sm">8 curated professional color schemes with instant preview</p>
                </div>
              </div>
              
              <div className="flex items-start space-x-3">
                <div className="w-6 h-6 bg-white/20 rounded-full flex items-center justify-center mt-0.5">
                  <span className="text-sm">✓</span>
                </div>
                <div>
                  <h3 className="font-semibold">Smart Avatar System</h3>
                  <p className="text-blue-100 text-sm">Initials, emojis, and pattern-based avatars with consistent generation</p>
                </div>
              </div>

              <div className="flex items-start space-x-3">
                <div className="w-6 h-6 bg-white/20 rounded-full flex items-center justify-center mt-0.5">
                  <span className="text-sm">✓</span>
                </div>
                <div>
                  <h3 className="font-semibold">Premium Preloader</h3>
                  <p className="text-blue-100 text-sm">Animated loading experience with progress tracking</p>
                </div>
              </div>
            </div>
            
            <div className="space-y-4">
              <div className="flex items-start space-x-3">
                <div className="w-6 h-6 bg-white/20 rounded-full flex items-center justify-center mt-0.5">
                  <span className="text-sm">✓</span>
                </div>
                <div>
                  <h3 className="font-semibold">Professional Image Placeholders</h3>
                  <p className="text-blue-100 text-sm">Context-aware placeholders for missing images</p>
                </div>
              </div>

              <div className="flex items-start space-x-3">
                <div className="w-6 h-6 bg-white/20 rounded-full flex items-center justify-center mt-0.5">
                  <span className="text-sm">✓</span>
                </div>
                <div>
                  <h3 className="font-semibold">Single Admin Workspace</h3>
                  <p className="text-blue-100 text-sm">Clean installation with only current user's workspace</p>
                </div>
              </div>

              <div className="flex items-start space-x-3">
                <div className="w-6 h-6 bg-white/20 rounded-full flex items-center justify-center mt-0.5">
                  <span className="text-sm">✓</span>
                </div>
                <div>
                  <h3 className="font-semibold">Professional Database Setup</h3>
                  <p className="text-blue-100 text-sm">Modern Blue theme applied by default during installation</p>
                </div>
              </div>
            </div>
          </div>
        </motion.section>
      </div>
    </div>
  )
}