"use client"

import { useEffect, useState } from "react"
import { motion, AnimatePresence } from "framer-motion"
import useTranslate from "@/hooks/useTranslate"

export default function PremiumPreloader({
  siteName = "Postuno",
  siteLogo,
  onComplete,
  primaryColor = "#667eea",
  secondaryColor = "#764ba2"
}) {
  const [loading, setLoading] = useState(true)
  const [progress, setProgress] = useState(0)

  const t = useTranslate()

  useEffect(() => {
    // Simple, fast progress animation
    const progressInterval = setInterval(() => {
      setProgress(prev => {
        if (prev >= 100) {
          clearInterval(progressInterval)
          setTimeout(() => {
            setLoading(false)
            if (onComplete) onComplete()
          }, 300) // Quick exit
          return 100
        }
        // Faster, more predictable progress
        return Math.min(prev + 3, 100)
      })
    }, 30)

    return () => clearInterval(progressInterval)
  }, [onComplete])

  return (
    <AnimatePresence>
      {loading && (
        <motion.div
          initial={{ opacity: 0 }}
          animate={{ opacity: 1 }}
          exit={{ opacity: 0 }}
          transition={{ duration: 0.3 }}
          className="fixed inset-0 z-[9999] bg-gradient-to-br from-white to-gray-50 dark:from-zinc-900 dark:to-zinc-950 flex flex-col items-center justify-center"
        >
          {/* Simple Logo */}
          <motion.div
            initial={{ scale: 0.8, opacity: 0 }}
            animate={{ scale: 1, opacity: 1 }}
            transition={{ duration: 0.4 }}
            className="mb-8"
          >
            {siteLogo ? (
              <img
                src={siteLogo}
                alt="Logo"
                className="w-16 h-16 object-contain"
              />
            ) : (
              <div
                className="w-16 h-16 rounded-xl flex items-center justify-center text-white text-2xl font-bold"
                style={{
                  background: `linear-gradient(135deg, ${primaryColor} 0%, ${secondaryColor} 100%)`
                }}
              >
                {siteName.charAt(0)}
              </div>
            )}
          </motion.div>

          {/* Site Name */}
          <motion.h1
            initial={{ y: 20, opacity: 0 }}
            animate={{ y: 0, opacity: 1 }}
            transition={{ delay: 0.2, duration: 0.4 }}
            className="text-2xl font-semibold text-gray-900 dark:text-white mb-8"
          >
            {siteName}
          </motion.h1>

          {/* Simple Loading Spinner */}
          <div className="relative mb-6">
            <motion.div
              animate={{ rotate: 360 }}
              transition={{
                duration: 1,
                repeat: Infinity,
                ease: "linear"
              }}
              className="w-8 h-8 border-2 border-gray-200 dark:border-zinc-700 rounded-full"
              style={{
                borderTopColor: primaryColor
              }}
            />
          </div>

          {/* Progress Bar */}
          <div className="w-64 max-w-sm mb-4">
            <div className="w-full bg-gray-200 dark:bg-zinc-800 rounded-full h-1.5 overflow-hidden">
              <motion.div
                initial={{ width: 0 }}
                animate={{ width: `${progress}%` }}
                transition={{ duration: 0.3, ease: "easeOut" }}
                className="h-full rounded-full"
                style={{
                  background: `linear-gradient(90deg, ${primaryColor} 0%, ${secondaryColor} 100%)`
                }}
              />
            </div>
          </div>

          {/* Loading Text */}
          <motion.p
            initial={{ opacity: 0 }}
            animate={{ opacity: 1 }}
            transition={{ delay: 0.4 }}
            className="text-sm text-gray-500 dark:text-gray-400"
          >
            {progress < 100 ? t("Loading...") : t("Almost ready!")}
          </motion.p>
        </motion.div>
      )}
    </AnimatePresence>
  )
}